use axum::{extract::FromRequestParts, http::request::Parts};
use sea_orm::DatabaseConnection;
use serde::Serialize;

use super::backend::AppState;
use crate::{
    error::{self, OwnError},
    fellowship::{Fellowship, request::FellowshipRequest},
    server::users::User,
};

pub mod adversary;
pub mod character;
pub mod enums;
pub mod equipment;
pub mod fellowship;
pub mod general;
pub mod message;
pub mod new_character;
pub mod socket;
pub mod tools;

#[derive(Serialize, Clone, Debug)]
pub enum LoremasterOrPlayer {
    Loremaster(User),
    Player(FellowshipRequest),
}

impl FromRequestParts<AppState> for LoremasterOrPlayer {
    type Rejection = OwnError;
    async fn from_request_parts(
        parts: &mut Parts,
        state: &AppState,
    ) -> Result<Self, Self::Rejection> {
        let user = if let Ok(x) = User::from_request_parts(parts, state).await {
            Self::Loremaster(x)
        } else {
            Self::Player(FellowshipRequest::from_request_parts(parts, state).await?)
        };
        Ok(user)
    }
}

impl LoremasterOrPlayer {
    pub fn get_loremaster(&self) -> User {
        match self {
            LoremasterOrPlayer::Loremaster(x) => x.clone(),
            LoremasterOrPlayer::Player(x) => x.loremaster.clone(),
        }
    }

    pub async fn get_fellowship(
        &self,
        db: &DatabaseConnection,
        fellowship_id: i32,
    ) -> error::Result<Fellowship> {
        let out = match self {
            LoremasterOrPlayer::Loremaster(x) => Fellowship::get(db, x, fellowship_id).await?,
            LoremasterOrPlayer::Player(x) => x.fellowship.clone(),
        };
        Ok(out)
    }
}
