use std::collections::HashMap;

use axum::Json;
use axum::extract::{Path, State};
use axum::routing::{delete, get, post, put};

use crate::adversary::{Adversary, AdversaryEnum};
use crate::error::Result;
use crate::server::backend::AppState;
use crate::server::cache_response::CacheResponse;
use crate::server::connection_handler::ConnectionHandler;
use crate::server::name_or_id::NameOrId;
use crate::server::users::User;

use super::socket::Event;

pub fn get_router() -> axum::Router<AppState> {
    axum::Router::new()
        .route("/official", get(get_official_adversaries))
        .route("/personal", get(get_personal_adversaries))
        .route("/{id}", get(get_adversary))
        .route("/{id}", delete(delete_adversary))
        .route("/", put(save_adversary))
        .route(
            "/{adversary_id}/loose-Endurance/{value}",
            post(loose_endurance),
        )
        .route("/{adversary_id}/loose-Hate/{value}", post(loose_hate))
        .route("/{adversary_id}/loose-Resolve/{value}", post(loose_resolve))
}

async fn get_official_adversaries() -> CacheResponse<HashMap<AdversaryEnum, Adversary>> {
    CacheResponse::new(Adversary::get_list_from_enums())
}

async fn get_personal_adversaries(
    state: State<AppState>,
    user: User,
) -> Result<Json<Vec<Adversary>>> {
    let advs = Adversary::get_list_from_db(&state.db, user).await?;
    Ok(Json(advs))
}

async fn get_adversary(
    state: State<AppState>,
    user: User,
    Path(id): Path<NameOrId>,
) -> Result<Json<Adversary>> {
    let adversary = Adversary::get(&state.db, &user, id).await?;
    Ok(Json(adversary))
}

async fn delete_adversary(state: State<AppState>, user: User, Path(id): Path<i32>) -> Result<()> {
    Adversary::delete_from_db(&state.db, user, id).await?;
    Ok(())
}

async fn save_adversary(
    state: State<AppState>,
    user: User,
    Json(adversary): Json<Adversary>,
) -> Result<Json<i32>> {
    let id = adversary.save_to_db(&state.db, &user).await?;
    Ok(Json(id))
}

// TODO merge all loose together (same for character)
async fn loose_endurance(
    state: State<AppState>,
    user: User,
    Path((adversary_id, value)): Path<(i32, i32)>,
) -> Result<()> {
    let adversary_id = NameOrId::Id(adversary_id);
    let mut adversary = Adversary::get(&state.db, &user, adversary_id).await?;

    adversary.loose_endurance(value);

    adversary.save_to_db(&state.db, &user).await?;

    // Update players
    ConnectionHandler::update_everyone(user.username.clone(), Event::UpdateFellowship).await;
    Ok(())
}

async fn loose_hate(
    state: State<AppState>,
    user: User,
    Path((adversary_id, value)): Path<(i32, i32)>,
) -> Result<()> {
    let adversary_id = NameOrId::Id(adversary_id);
    let mut adversary = Adversary::get(&state.db, &user, adversary_id).await?;

    adversary.loose_hate(value);

    adversary.save_to_db(&state.db, &user).await?;

    // Update players
    ConnectionHandler::update_everyone(user.username.clone(), Event::UpdateFellowship).await;
    Ok(())
}

async fn loose_resolve(state: State<AppState>, user: User, path: Path<(i32, i32)>) -> Result<()> {
    loose_hate(state, user, path).await
}
