use serde::{Deserialize, Deserializer, de::Error as SerdeError};
use serde_json::Value;
use tracing::error;

#[derive(PartialEq, Debug)]
pub enum NameOrId {
    Id(i32),
    Name(String),
}

impl<'de> Deserialize<'de> for NameOrId {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        let data: Value = Deserialize::deserialize(deserializer)?;
        Ok(match data {
            Value::String(data) => {
                if let Ok(id) = data.parse::<i32>() {
                    NameOrId::Id(id)
                } else {
                    NameOrId::Name(data)
                }
            }
            Value::Number(x) => NameOrId::Id(
                x.as_i64()
                    .ok_or(SerdeError::custom("Failed to convert number to i64"))?
                    .try_into()
                    .map_err(|err| {
                        error!(?err);
                        SerdeError::custom("Failed to convert i64 to i32")
                    })?,
            ),
            _ => Err(SerdeError::custom("Failed to deserialize NameOrId"))?,
        })
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    #[allow(clippy::restriction)]
    fn deserialization() -> anyhow::Result<()> {
        #[derive(Deserialize)]
        struct Test {
            value: NameOrId,
        }

        assert_eq!(
            serde_json::from_str::<Test>("{\"value\": 32}")?.value,
            NameOrId::Id(32)
        );
        assert_eq!(
            serde_json::from_str::<Test>("{\"value\": \"Name\"}")?.value,
            NameOrId::Name("Name".to_owned())
        );

        Ok(())
    }
}
