use std::str::from_utf8;

use rust_embed::Embed;
use serde::{Deserialize, Serialize};

#[derive(Embed)]
#[folder = "data"]
pub struct Assets;

pub fn read_enum_from_assets<Enum, Type>(value: &Enum, directory: &str) -> Type
where
    Enum: Serialize,
    Type: for<'a> Deserialize<'a>,
{
    let mut value = serde_json::to_string(&value).expect("Failed to serialize");

    // Remove \"\"
    value.pop();
    value.remove(0);

    let filename = format!("{directory}/{value}.json");

    read_file(&filename)
}

pub fn read_file<Type>(filename: &str) -> Type
where
    Type: for<'a> Deserialize<'a>,
{
    let file = Assets::get(filename).expect("Failed to open asset");
    let content = from_utf8(&file.data).expect("Failed to convert to &str");
    serde_json::from_str(content).expect("Failed to deserialize")
}

#[cfg(test)]
pub fn get_number_files(directory: &str) -> usize {
    let mut count = 0;
    for file in Assets::iter() {
        if file.contains(directory) {
            count += 1;
        }
    }
    count
}

#[cfg(test)]
mod test {
    use anyhow::bail;
    use serde_json::Value;
    use std::{collections::BTreeMap, env, fs};

    use super::*;

    const TRANSLATED_FIELDS: [&str; 12] = [
        "description",
        "title",
        "name",
        "positive",
        "negative",
        "distinctive_features",
        "shadow_path",
        "advantage",
        "weapon",
        "special_damages",
        "features",
        "distinctive_feature",
    ];

    fn process_data(key: &str, value: &Value) -> Vec<String> {
        match value {
            Value::String(text) => {
                if TRANSLATED_FIELDS.contains(&key) {
                    vec![text.clone()]
                } else {
                    vec![]
                }
            }
            Value::Array(array) => array
                .iter()
                .flat_map(|value| process_data(key, value))
                .collect(),
            Value::Object(object) => object
                .iter()
                .flat_map(|(key, value)| process_data(key, value))
                .collect(),
            _ => vec![],
        }
    }

    #[test]
    fn generate_translation() -> anyhow::Result<()> {
        let translation_file = env::var("TRANSLATION_FILE")?;
        let translation_data = fs::read_to_string(&translation_file)?;
        let translation: BTreeMap<String, String> = serde_json::from_str(&translation_data)?;
        let mut missing = false;

        for filename in Assets::iter() {
            println!("{filename:?}");
            let data = read_file::<Value>(&filename);
            let texts = process_data("root", &data);
            for text in texts {
                if !translation.contains_key(&text) {
                    println!("Missing {text:?}");
                    missing = true;
                }
            }
        }

        if missing {
            bail!("Translation has missing items");
        }
        Ok(())
    }
}
