import { type NarviCharacter } from '../narvi/bindings/NarviCharacter';
import { type Armor } from '../the_one_app/bindings/Armor';
import { type CallingEnum } from '../the_one_app/bindings/CallingEnum';
import { type Character } from '../the_one_app/bindings/Character';
import { type CombatProficiency } from '../the_one_app/bindings/CombatProficiency';
import { type CultureEnum } from '../the_one_app/bindings/CultureEnum';
import { type Equipment } from '../the_one_app/bindings/Equipment';
import { type Reward } from '../the_one_app/bindings/Reward';
import { type Shield } from '../the_one_app/bindings/Shield';
import { type SkillEnum } from '../the_one_app/bindings/SkillEnum';
import { type Virtue } from '../the_one_app/bindings/Virtue';
import { type Weapon } from '../the_one_app/bindings/Weapon';
import { camel_case } from './utils';

function concatenate(values: (string | undefined)[]): string {
    return values.filter((x) => x != undefined && x.length != 0).join(', ');
}

function extract_skill_value(value: string): number {
    return Number(value.replace('f', ''));
}

function get_favoured_skills(narvi: NarviCharacter): SkillEnum[] {
    const skill_enums: SkillEnum[] = [
        'Awe',
        'Athletics',
        'Awareness',
        'Hunting',
        'Song',
        'Craft',
        'Enhearten',
        'Travel',
        'Insight',
        'Healing',
        'Courtesy',
        'Battle',
        'Persuade',
        'Stealth',
        'Scan',
        'Explore',
        'Riddle',
        'Lore'
    ];
    return skill_enums
        .map((x): [SkillEnum, string] => {
            const attribute = `skill${x.toLowerCase()}`;
            if (!(attribute in narvi)) {
                throw 'Attribute does not exist';
            }
            /* @ts-expect-error not typed */
            const value: string = String(narvi[attribute]);
            return [x, value];
        })
        .filter((x) => x[1] == 'TRUE')
        .map((x) => x[0]);
}

function get_shields(narvi: NarviCharacter, index: number[]): Shield[] {
    return index
        .filter((x) => {
            // @ts-expect-error not typed
            const name = narvi[`armour${x}name`];
            // @ts-expect-error not typed
            const bonus = Number(narvi[`armour${x}bonus`]);
            return name != undefined && name.length != 0 && bonus != 0;
        })
        .map((x) => {
            // @ts-expect-error not typed
            const wearing = narvi[`armour${x}equipped`] == 'TRUE';
            // @ts-expect-error not typed
            const name = narvi[`armour${x}name`];
            // @ts-expect-error not typed
            const parry = Number(narvi[`armour${x}bonus`]);
            // @ts-expect-error not typed
            const load = Number(narvi[`armour${x}enc`]);
            const notes = get_armor_note(narvi, x);
            return {
                famous: false,
                id: null,
                load: { value: load, computed_value: 0 },
                name: name,
                notes: notes,
                parry: { value: parry, computed_value: 0 },
                rewards: [],
                wearing: wearing,
                enum_value: null
            };
        });
}

function get_armor_note(narvi: NarviCharacter, index: number): string {
    // @ts-expect-error not typed
    const origin = narvi[`armour${index}origin`];
    // @ts-expect-error not typed
    const q1 = narvi[`armour${index}quality1`];
    // @ts-expect-error not typed
    const q2 = narvi[`armour${index}quality2`];
    // @ts-expect-error not typed
    const q3 = narvi[`armour${index}quality3`];
    // @ts-expect-error not typed
    const b1 = narvi[`armour${index}bane1`];
    // @ts-expect-error not typed
    const b2 = narvi[`armour${index}bane2`];
    // @ts-expect-error not typed
    const curse = narvi[`armour${index}curse`];
    return concatenate([origin, q1, q2, q3, b1, b2, curse]);
}

function get_armors(narvi: NarviCharacter, index: number[]): Armor[] {
    return index
        .filter((x) => {
            // @ts-expect-error not typed
            const name = narvi[`armour${x}name`];
            // @ts-expect-error not typed
            const bonus = Number(narvi[`armour${x}bonus`]);
            return name != undefined && name.length != 0 && bonus == 0;
        })
        .map((x) => {
            // @ts-expect-error not typed
            const wearing = narvi[`armour${x}equipped`] == 'TRUE';
            // @ts-expect-error not typed
            const name = narvi[`armour${x}name`];
            // @ts-expect-error not typed
            const protection = Number(narvi[`armour${x}protect`]);
            // @ts-expect-error not typed
            const load = Number(narvi[`armour${x}enc`]);
            const notes = get_armor_note(narvi, x);
            return {
                famous: false,
                id: null,
                load: { value: load, computed_value: 0 },
                name: name,
                notes: notes,
                protection: protection,
                rewards: [],
                wearing: wearing,
                enum_value: null
            };
        });
}

function get_useful_items(narvi: NarviCharacter, index: number[]): Equipment[] {
    return index
        .filter((x) => {
            // @ts-expect-error not typed
            const name = narvi[`usefulitem${x}`];
            return name != undefined && name.length != 0;
        })
        .map((x) => {
            // @ts-expect-error not typed
            const name = narvi[`usefulitem${x}`];
            // @ts-expect-error not typed
            const skill = narvi[`usefulitem${x}skill`];
            return {
                name: name,
                famous: false,
                id: null,
                load: { value: 0, computed_value: 0 },
                notes: skill != undefined && skill.length != 0 ? skill : '',
                rewards: [],
                wearing: true,
                skill: null
            };
        });
}

function get_other_items(narvi: NarviCharacter, index: number[]): Equipment[] {
    return index
        .filter((x) => {
            // @ts-expect-error not typed
            const name = narvi[`otheritem${x}`];
            return name != undefined && name.length != 0;
        })
        .map((x) => {
            // @ts-expect-error not typed
            const name = narvi[`otheritem${x}`];
            return {
                name: name,
                famous: false,
                id: null,
                load: { value: 0, computed_value: 0 },
                notes: '',
                rewards: [],
                wearing: true,
                skill: null
            };
        });
}

function get_marvelous_items(narvi: NarviCharacter, index: number[]): Equipment[] {
    return index
        .filter((x) => {
            // @ts-expect-error not typed
            const name = narvi[`marvellous${x}name`];
            return name != undefined && name.length != 0;
        })
        .map((x) => {
            // @ts-expect-error not typed
            const name = narvi[`marvellous${x}name`];
            // @ts-expect-error not typed
            const b1 = narvi[`marvellous${x}blessing1`];
            // @ts-expect-error not typed
            const b2 = narvi[`marvellous${x}blessing2`];
            return {
                name: name,
                famous: false,
                id: null,
                load: { value: 0, computed_value: 0 },
                notes: concatenate([b1, b2]),
                rewards: [],
                wearing: true,
                skill: null
            };
        });
}

function get_rewards(narvi_values: string[]): Reward[] {
    return narvi_values
        .filter((x) => x != undefined && x.length != 0)
        .map((x) => {
            return {
                available_at_creation: false,
                description: '',
                enum_value: null,
                id: null,
                linked_item: null,
                title: x,
                implemented: false
            };
        });
}

function get_virtues(narvi_values: string[]): Virtue[] {
    return narvi_values
        .filter((x) => x != undefined && x.length != 0)
        .map((x) => {
            return {
                available_at_creation: false,
                description: '',
                enum_value: null,
                id: null,
                title: x,
                available: false,
                implemented: false,
                restricted_to: [],
                text_input: ''
            };
        });
}

function get_culture(culture: string): CultureEnum {
    switch (culture) {
        case 'Bree-Hobbit':
            return 'HobbitOfBree';
        case 'DwarfOfNogrodAndBelegost':
            return 'DwarfOfNogrodOrBelegost';
        case 'Bree-folk':
            return 'ManOfBree';
        default:
            return camel_case(culture) as CultureEnum;
    }
}

function get_weapons(narvi: NarviCharacter, index: number[]): Weapon[] {
    return index
        .filter((x) => {
            // @ts-expect-error not typed
            const name = narvi[`weapon${x}name`];
            return name != undefined && name.length != 0;
        })
        .map((x) => {
            // @ts-expect-error not typed
            const wearing = narvi[`weapon${x}equipped`] == 'TRUE';
            // @ts-expect-error not typed
            const name = narvi[`weapon${x}name`];
            // @ts-expect-error not typed
            const damage = Number(narvi[`weapon${x}dmg`]);
            // @ts-expect-error not typed
            const injury = Number(narvi[`weapon${x}injury`]);
            // @ts-expect-error not typed
            const proficiency = narvi[`weapon${x}proficiency`] as CombatProficiency;
            // @ts-expect-error not typed
            const load = Number(narvi[`weapon${x}enc`]);
            // @ts-expect-error not typed
            const notes = narvi[`weapon${x}notes`];
            // @ts-expect-error not typed
            const origin = narvi[`weapon${x}origin`];
            // @ts-expect-error not typed
            const q1 = narvi[`weapon${x}quality1`];
            // @ts-expect-error not typed
            const q2 = narvi[`weapon${x}quality2`];
            // @ts-expect-error not typed
            const q3 = narvi[`weapon${x}quality3`];
            // @ts-expect-error not typed
            const b1 = narvi[`weapon${x}bane1`];
            // @ts-expect-error not typed
            const b2 = narvi[`weapon${x}bane2`];
            // @ts-expect-error not typed
            const curse = narvi[`weapon${x}curse`];

            return {
                name: name,
                damage: { value: damage, computed_value: 0 },
                famous: false,
                id: null,
                combat_proficiency: proficiency,
                injury: { value: injury, computed_value: 0 },
                two_hand_injury: null,
                load: { value: load, computed_value: 0 },
                notes: concatenate([notes, origin, q1, q2, q3, b1, b2, curse]),
                wearing: wearing,
                rewards: [],
                enum_value: null
            };
        });
}

export function narvi_to_the_one_app(narvi: NarviCharacter): Character {
    return {
        name: narvi.name,
        age: Number(narvi.age),
        calling: camel_case(narvi.calling) as CallingEnum,
        char_type: 'Internal',
        characteristics: {
            adventure_experience: {
                current: Number(narvi.adventurepointsavailable),
                total: Number(narvi.adventurepointstotal)
            },
            attributes: {
                endurance: {
                    current: Number(narvi.endurance),
                    max: Number(narvi.maxendurance),
                    computed_max: 0
                },
                fatigue: Number(narvi.fatigue),
                heart: {
                    value: Number(narvi.heart),
                    derived: {
                        target: 0
                    }
                },
                hope: {
                    current: Number(narvi.hope),
                    max: Number(narvi.maxhope),
                    computed_max: 0
                },
                parry: { value: Number(narvi.parry), computed_value: 0 },
                shadow: {
                    current: Number(narvi.shadow),
                    derived: {
                        total_shadow: 0
                    },
                    path: narvi.shadowpath,
                    scars: Number(narvi.shadowscars)
                },
                strength: {
                    value: Number(narvi.strength),
                    derived: {
                        target: 0
                    }
                },
                wits: {
                    value: Number(narvi.wits),
                    derived: {
                        target: 0
                    }
                }
            },
            combat_skills: {
                skills: [narvi.axes, narvi.bows, narvi.spears, narvi.swords]
            },
            skill_experience: {
                current: Number(narvi.skillpointsavailable),
                total: Number(narvi.skillpointstotal)
            },
            skills: {
                skills: [
                    extract_skill_value(narvi.awe),
                    extract_skill_value(narvi.athletics),
                    extract_skill_value(narvi.awareness),
                    extract_skill_value(narvi.hunting),
                    extract_skill_value(narvi.song),
                    extract_skill_value(narvi.craft),
                    extract_skill_value(narvi.enhearten),
                    extract_skill_value(narvi.travel),
                    extract_skill_value(narvi.insight),
                    extract_skill_value(narvi.healing),
                    extract_skill_value(narvi.courtesy),
                    extract_skill_value(narvi.battle),
                    extract_skill_value(narvi.persuade),
                    extract_skill_value(narvi.stealth),
                    extract_skill_value(narvi.scan),
                    extract_skill_value(narvi.explore),
                    extract_skill_value(narvi.riddle),
                    extract_skill_value(narvi.lore)
                ],
                favoured: get_favoured_skills(narvi)
            },
            valor: {
                value: Number(narvi.valour),
                advantages: get_rewards([
                    narvi.reward1,
                    narvi.reward2,
                    narvi.reward3,
                    narvi.reward4,
                    narvi.reward5,
                    narvi.reward6
                ])
            },
            wisdom: {
                value: Number(narvi.wisdom),
                advantages: get_virtues([
                    narvi.cultureblessing,
                    narvi.virtue1,
                    narvi.virtue2,
                    narvi.virtue3,
                    narvi.virtue4,
                    narvi.virtue5,
                    narvi.virtue6,
                    narvi.virtue7,
                    narvi.virtue8
                ])
            }
        },
        culture: get_culture(narvi.culture),
        derived: {
            total_load_fatigue: 0,
            total_parry: 0
        },
        distinctive_features: [narvi.feature1, narvi.feature2, narvi.feature3],
        flaws: [],
        id: null,
        injury: {
            description: '',
            duration: Number(narvi.wounddaysleft)
        },
        items: {
            derived: {
                load: 0,
                parry: 0,
                protection: 0
            },
            treasure: {
                carrying: Number(narvi.carriedtreasure),
                total: Number(narvi.treasure),
                derived: {
                    standard_of_living: 'Frugal'
                }
            },
            armors: get_armors(narvi, [1, 2, 3, 4]),
            equipments: [
                ...get_useful_items(narvi, [1, 2, 3, 4]),
                ...get_other_items(narvi, [1, 2, 3]),
                ...get_marvelous_items(narvi, [1, 2, 3, 4, 5, 6, 7, 8])
            ],
            weapons: get_weapons(narvi, [1, 2, 3, 4]),
            shields: get_shields(narvi, [1, 2, 3, 4])
        },
        notes: '',
        one_shot_rules: false,
        player_name: '',
        strider_mode: false
    };
}
