use std::{fs::File, io::Write, path::PathBuf};

use clap::Parser;
use enums::{fell_ability::FellAbilityEnum, reward::RewardEnum};
use serde::Deserialize;
use strum::IntoEnumIterator;
use tools::{get_translations, write_translations};

#[derive(Deserialize, Parser, Clone)]
struct Flags {
    #[clap(subcommand)]
    tp: DataType,
}

#[derive(Parser, Deserialize, Clone)]
enum DataType {
    FellAbilities,
    Rewards,
}

fn main() {
    const ROOT_PATH: &str = "data";
    let args = Flags::parse();

    let mut translations = get_translations();

    match args.tp {
        DataType::FellAbilities => FellAbilityEnum::iter().for_each(|ability| {
            let ability_str = format!("{ability:?}");
            let ability_descr = format!("{ability:?}_description");
            let filename = PathBuf::from(ROOT_PATH)
                .join("adversaries")
                .join("fell_abilities")
                .join(&ability_str)
                .with_extension("json");

            let write_to_file = !filename.is_file();
            let write_to_translation = !translations.contains_key(&ability_str);
            if !write_to_file && !write_to_translation {
                println!("     Skipping {ability:?}");
                return;
            }

            if write_to_file {
                println!("Writing {ability:?}");
                let mut file = File::create(filename).expect("Failed to create file");
                write!(
                    file,
                    "{{\"description\": \"{ability_descr}\", \"title\": \"{ability:?}\"}}"
                )
                .expect("Failed to write to file");
            }

            if write_to_translation {
                println!("Translating {ability:?}");
                translations.insert(ability_str, String::from("TODO"));
                translations.insert(ability_descr, String::from("TODO"));
            }
        }),
        DataType::Rewards => RewardEnum::iter().for_each(|reward| {
            let reward_str = format!("{reward:?}");
            let reward_descr = format!("{reward:?}_description");
            let filename = PathBuf::from(ROOT_PATH)
                .join("rewards")
                .join(&reward_str)
                .with_extension("json");

            let write_to_file = !filename.is_file();
            let write_to_translation = !translations.contains_key(&reward_str);
            if !write_to_file && !write_to_translation {
                println!("     Skipping {reward:?}");
                return;
            }

            if write_to_file {
                println!("Writing {reward:?}");
                let mut file = File::create(filename).expect("Failed to create file");
                write!(
                    file,
                    "{{\"description\": \"{reward_descr}\", \"title\": \"{reward:?}\", \"available_at_creation\": false}}"
                )
                .expect("Failed to write to file");
            }

            if write_to_translation {
                println!("Translating {reward:?}");
                translations.insert(reward_str, String::from("TODO"));
                translations.insert(reward_descr, String::from("TODO"));
            }
        }),
    }

    write_translations(&translations);
}
