use std::collections::HashSet;

use anyhow::{Result, bail};
use enums::{culture::CultureEnum, virtue::VirtueEnum};
use sea_orm::{
    ActiveModelTrait as _,
    ActiveValue::{NotSet, Set},
    ColumnTrait as _, ConnectionTrait, EntityTrait as _, ModelTrait as _, QueryFilter as _,
};
use serde::{Deserialize, Serialize, de};
use ts_rs::TS;

use crate::entity::virtue as entity;
use crate::utils::convert_option_to_set;
use crate::{assets::read_enum_from_assets, utils::to_enum};

use super::advantage::AdvantageTrait;

const DIRECTORY: &str = "virtues";

#[derive(Serialize, Deserialize, Clone, Default, TS, Debug)]
#[ts(export)]
pub struct Virtue {
    id: Option<i32>,
    title: String,
    #[serde(default)]
    enum_value: Option<VirtueEnum>,
    description: String,
    #[serde(default)]
    text_input: String,
    available: bool,
    available_at_creation: bool,
    #[serde(default, deserialize_with = "deserialize_restricted_to")]
    restricted_to: Option<HashSet<CultureEnum>>,
    #[serde(default)]
    implemented: bool,
}

// AdvantageTrait
impl AdvantageTrait for Virtue {
    type Enum = VirtueEnum;
    fn get_title(&self) -> &str {
        &self.title
    }
    fn set_id(&mut self, id: Option<i32>) {
        self.id = id;
    }

    fn get_id(&self) -> Option<i32> {
        self.id
    }

    fn from_enum(value: Self::Enum) -> Self {
        let mut out: Self = read_enum_from_assets(&value, DIRECTORY);
        out.enum_value = Some(value);
        out
    }

    fn get_enum(&self) -> Option<Self::Enum> {
        self.enum_value
    }

    fn find_enum(&mut self) -> bool {
        // TODO remove (compatibility)
        let title = to_enum(&self.title);
        let title = format!("\"{title}\"");
        if let Ok(out) = serde_json::from_str(&title) {
            self.enum_value = Some(out);
            true
        } else {
            false
        }
    }

    fn is_available(&self) -> bool {
        self.available
    }

    fn is_available_at_creation(&self) -> bool {
        self.available_at_creation
    }

    async fn from_db_model<C: ConnectionTrait>(db: &C, character_id: i32) -> Result<Vec<Self>> {
        Ok(entity::Entity::find()
            .filter(entity::Column::CharacterId.eq(character_id))
            .all(db)
            .await?
            .iter()
            .map(|x| (*x).clone().into())
            .collect())
    }

    async fn save<C: ConnectionTrait>(&self, db: &C, character_id: i32) -> Result<()> {
        let mut model: entity::ActiveModel = (*self).clone().into();
        model.character_id = Set(character_id);
        model.save(db).await?;
        Ok(())
    }

    async fn delete_by_character_id<C: ConnectionTrait>(db: &C, character_id: i32) -> Result<()> {
        entity::Entity::delete_many()
            .filter(entity::Column::CharacterId.eq(character_id))
            .exec(db)
            .await?;
        Ok(())
    }

    async fn delete_by_id<C: ConnectionTrait>(db: &C, character_id: i32, id: i32) -> Result<()> {
        let virtue = entity::Entity::find_by_id(id).one(db).await?;

        if let Some(virtue) = virtue {
            if virtue.character_id == character_id {
                virtue.delete(db).await?;
                Ok(())
            } else {
                bail!("Cannot delete virtue from someone else")
            }
        } else {
            bail!("Virtue not found")
        }
    }
}

// Implementation
impl Virtue {
    pub fn get_culture_restrictions(&self) -> Option<HashSet<CultureEnum>> {
        self.restricted_to.clone()
    }
    pub fn is_implemented(&self) -> bool {
        self.implemented
    }
    pub fn is(&self, value: VirtueEnum) -> bool {
        match self.enum_value {
            Some(x) => x == value,
            None => false,
        }
    }
}

// TODO delete this (compatibility)
#[derive(Deserialize)]
#[serde[untagged]]
enum RestrictedTo {
    V1(Option<CultureEnum>),
    V2(Option<HashSet<CultureEnum>>),
}

// TODO delete this (compatibility)
fn deserialize_restricted_to<'de, D>(
    deserializer: D,
) -> Result<Option<HashSet<CultureEnum>>, D::Error>
where
    D: de::Deserializer<'de>,
{
    let rest: RestrictedTo = de::Deserialize::deserialize(deserializer)?;
    match rest {
        RestrictedTo::V2(x) => Ok(x),
        RestrictedTo::V1(x) => {
            if let Some(x) = x {
                let mut hash = HashSet::new();
                hash.insert(x);
                Ok(Some(hash))
            } else {
                Ok(None)
            }
        }
    }
}

impl From<entity::Model> for Virtue {
    fn from(value: entity::Model) -> Self {
        let enum_value = value.enum_value.as_ref().and_then(|name| {
            serde_json::from_str(name).expect("SQL and serde enums are not compatible")
        });

        let mut out = Self {
            id: Some(value.id),
            title: value.title,
            enum_value,
            description: value.description,
            text_input: value.text_input,
            available: false,
            available_at_creation: false,
            restricted_to: None,
            implemented: false,
        };
        if enum_value.is_none() {
            out.find_enum();
        }
        let enum_value = out.get_enum();
        if let Some(enum_value) = enum_value {
            out.implemented = Self::from_enum(enum_value).implemented;
        }
        out
    }
}

impl From<Virtue> for entity::ActiveModel {
    fn from(value: Virtue) -> Self {
        let id = convert_option_to_set(value.id);
        let enum_value = value
            .enum_value
            .as_ref()
            .map(|value| serde_json::to_string(value).expect("Failed to convert enum to string"));
        Self {
            id,
            title: Set(value.title),
            enum_value: Set(enum_value),
            description: Set(value.description),
            text_input: Set(value.text_input),
            character_id: NotSet,
        }
    }
}

// Testing
#[cfg(test)]
mod test {
    use strum::{EnumCount as _, IntoEnumIterator as _};

    use crate::{
        assets::get_number_files,
        character::{
            advantage::AdvantageTrait as _,
            virtue::{DIRECTORY, Virtue},
        },
    };

    use super::VirtueEnum;

    #[test]
    fn check_virtue_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(VirtueEnum::COUNT == number_files);

        // Check that all the known virtues work
        for virtue in VirtueEnum::iter() {
            eprintln!("Testing {virtue:?}");
            let read = Virtue::from_enum(virtue);
            let enum_read = read
                .get_enum()
                .expect("Official virtues should have their enum set");
            assert_eq!(enum_read, virtue);
        }
    }
    #[test]
    fn find_enum() {
        let mut virtue = Virtue {
            title: "Cram".to_owned(),
            ..Default::default()
        };
        virtue.find_enum();
        assert_eq!(virtue.enum_value, Some(VirtueEnum::Cram));
    }
}
