use std::vec;

use enums::character_data::CharacterData;
use enums::reward::RewardEnum;
use enums::shield::ShieldEnum;
use sea_orm::ActiveValue::{NotSet, Set};
use serde::{Deserialize, Serialize};
use ts_rs::TS;

use crate::assets::read_enum_from_assets;
use crate::character::reward::Reward;
use crate::common::computed_value::ComputedValue;
use crate::common::enums::find_enum;
use crate::entity::reward_shield;
use crate::entity::shield as entity;
use crate::impl_item_trait;
use crate::utils::convert_option_to_set;

const DIRECTORY: &str = "items/shields";

#[derive(Serialize, Deserialize, Clone, Default, TS, Debug)]
#[ts(export)]
pub struct Shield {
    id: Option<i32>,
    name: String,
    parry: ComputedValue<u32>,
    enum_value: Option<ShieldEnum>,
    load: ComputedValue<u32>,
    #[serde(default)]
    wearing: bool,
    #[serde(default)]
    notes: String,
    #[serde(default)]
    famous: bool,
    #[serde(default)]
    rewards: Vec<Reward>,
}

impl Shield {
    pub fn new(name: String, parry: u32, load: u32) -> Self {
        Self {
            name,
            parry: ComputedValue::new(parry),
            load: ComputedValue::new(load),
            wearing: true,
            ..Self::default()
        }
    }

    pub fn get_parry(&self) -> ComputedValue<u32> {
        self.parry
    }
    #[deprecated(since = "3.0.0", note = "Please use the sql functions")]
    pub fn reset_id(&mut self) {
        self.id = None;
    }
    fn compute_special_derived_values(&mut self, _data: &CharacterData) {
        let load_diff = if self
            .rewards
            .iter()
            .any(|rew| rew.get_enum() == Some(RewardEnum::MithrilShield))
            && let Some(shield) = self.enum_value
        {
            match shield {
                ShieldEnum::Buckler => 2,
                ShieldEnum::Shield | ShieldEnum::GreatShield => 4,
            }
        } else {
            0
        };
        self.load
            .set_computed_value(max(0, self.load.value - load_diff));
    }
}

impl From<ShieldEnum> for Shield {
    fn from(value: ShieldEnum) -> Self {
        let mut out: Self = read_enum_from_assets(&value, DIRECTORY);
        out.enum_value = Some(value);
        out
    }
}

impl From<&str> for Shield {
    fn from(value: &str) -> Self {
        let value = format!("\"{value}\"");
        let value: ShieldEnum =
            serde_json::from_str(&value).expect("Failed to convert to shield enum");
        Self::from(value)
    }
}

impl_item_trait!(
    Shield,
    reward_shield::Entity,
    entity::Entity,
    shield_id,
    reward_shield::ActiveModel
);

impl From<entity::Model> for Shield {
    fn from(value: entity::Model) -> Self {
        let enum_value = value.enum_value.as_ref().and_then(|name| {
            serde_json::from_str(name).expect("SQL and serde enums are not compatible")
        });
        Self {
            id: Some(value.id),
            enum_value: match enum_value {
                Some(x) => Some(x),
                None => find_enum(&value.name),
            },
            name: value.name,
            parry: ComputedValue::new(value.parry.try_into().expect("Failed to convert")),
            load: ComputedValue::new(value.load.try_into().expect("Failed to convert")),
            wearing: value.wearing,
            notes: value.notes,
            famous: value.famous,
            rewards: vec![],
        }
    }
}

impl From<Shield> for entity::ActiveModel {
    fn from(value: Shield) -> Self {
        let id = convert_option_to_set(value.id);
        let enum_value = value
            .enum_value
            .as_ref()
            .map(|value| serde_json::to_string(value).expect("Failed to convert enum to string"));
        Self {
            id,
            name: Set(value.name),
            parry: Set(value.parry.value.try_into().expect("Failed to convert")),
            load: Set(value.load.value.try_into().expect("Failed to convert")),
            wearing: Set(value.wearing),
            notes: Set(value.notes),
            character_id: NotSet,
            famous: Set(value.famous),
            enum_value: Set(enum_value),
        }
    }
}

#[cfg(test)]
mod test {
    use strum::{EnumCount as _, IntoEnumIterator as _};

    use crate::assets::get_number_files;

    use super::*;

    #[test]
    fn check_shield_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(ShieldEnum::COUNT == number_files);

        for shield in ShieldEnum::iter() {
            eprintln!("Testing {shield:?}");
            let _: Shield = Shield::from(shield);
        }
    }
}
