use std::fmt::Debug;

use enums::character_data::CharacterData;
use enums::skill::SkillEnum;
use sea_orm::{ActiveValue::NotSet, Set};
use serde::{Deserialize, Serialize};
use ts_rs::TS;

use crate::character::reward::Reward;
use crate::common::computed_value::ComputedValue;
use crate::entity::{equipment as entity, reward_equipment};
use crate::impl_item_trait;
use crate::utils::convert_option_to_set;

#[derive(Serialize, Deserialize, Clone, TS, Debug)]
#[ts(export)]
pub struct Equipment {
    id: Option<i32>,
    name: String,
    load: ComputedValue<u32>,
    wearing: bool,
    notes: String,
    famous: bool,
    rewards: Vec<Reward>,
    skill: Option<SkillEnum>,
}

impl Equipment {
    pub fn new(name: String, load: u32) -> Self {
        Self {
            id: None,
            name,
            load: ComputedValue::new(load),
            wearing: true,
            notes: String::new(),
            famous: false,
            rewards: vec![],
            skill: None,
        }
    }
    #[deprecated(since = "3.0.0", note = "Please use the sql functions")]
    pub fn reset_id(&mut self) {
        self.id = None;
    }
    #[expect(clippy::unused_self)]
    fn compute_special_derived_values(&self, _data: &CharacterData) {}
}

impl From<entity::Model> for Equipment {
    fn from(value: entity::Model) -> Self {
        let skill = value.skill.map(|skill| {
            serde_json::from_str(&skill).expect("SQL and serde enums are not compatible")
        });
        Self {
            id: Some(value.id),
            name: value.name,
            load: ComputedValue::new(value.load.try_into().expect("Failed to convert")),
            wearing: value.wearing,
            notes: value.notes,
            famous: value.famous,
            rewards: vec![],
            skill,
        }
    }
}

impl From<Equipment> for entity::ActiveModel {
    fn from(value: Equipment) -> Self {
        let id = convert_option_to_set(value.id);
        let skill = value
            .skill
            .map(|skill| serde_json::to_string(&skill).expect("Failed to convert enum to string"));
        Self {
            id,
            name: Set(value.name),
            load: Set(value.load.value.try_into().expect("Failed to convert")),
            wearing: Set(value.wearing),
            notes: Set(value.notes),
            character_id: NotSet,
            famous: Set(value.famous),
            skill: Set(skill),
        }
    }
}

impl_item_trait!(
    Equipment,
    reward_equipment::Entity,
    entity::Entity,
    equipment_id,
    reward_equipment::ActiveModel
);
