import { tauri } from '$lib/tauri.svelte';
import { data } from '$lib/utils.svelte';
import { fetch as tauri_fetch } from '@tauri-apps/plugin-http';
import { sleep } from 'the-one-app-api/utils/utils';
import { BackendServer } from 'the-one-app-api/the_one_app/queries';
import { toast } from '@zerodevx/svelte-toast';
import { get_socket } from './socket.svelte';
import { SocketMessage } from 'the-one-app-api/the_one_app/socket';

let backend: null | BackendServer = $state(null);
let fellowship_backend: null | BackendServer = $state(null);

export function set_backend(server: BackendServer | null) {
    backend = server;
}

export function set_fellowship_backend(server: BackendServer | null) {
    fellowship_backend = server;
}

function is_dev(): boolean {
    return import.meta.env.DEV;
}

export function get_use_auth(): boolean {
    if (backend == null) {
        return true;
    } else {
        return backend.use_auth;
    }
}

export function get_username(): string | null {
    if (backend == null || backend.auth == null) {
        return null;
    } else {
        return backend.auth.get_username();
    }
}

export async function get_with_wait<T>(get_variable: () => T | null, name: string): Promise<T> {
    const max_wait = 2000;
    let time = 0;
    const wait = 100;
    while (get_variable() == null && time < max_wait) {
        await sleep(wait);
        time += wait;
    }
    const out = get_variable();
    if (out == null) {
        const text = `${name} is still null`;
        toast.push(text);
        throw new Error(text);
    }
    return out;
}

export async function get_fellowship_backend(): Promise<BackendServer> {
    return get_with_wait(() => fellowship_backend, 'Fellowship backend');
}

export async function get_backend(): Promise<BackendServer> {
    return get_with_wait(() => backend, 'Backend');
}

export async function init_backend() {
    console.log('Init tauri');
    await tauri.init();
    console.log('Init data');
    await data.init();
    console.log('Backend has been initialized');
}

export async function get_server(): Promise<URL> {
    await tauri.wait_tauri();
    if (tauri.is_tauri) {
        return new URL('http://localhost:8000');
    } else if (is_dev()) {
        return new URL('http://localhost:8000');
    } else {
        return new URL(window.location.origin);
    }
}

export async function own_fetch(
    this: BackendServer,
    path: string,
    args: RequestInit | undefined
): Promise<Response> {
    await tauri.wait_tauri();
    const url = new URL(path, this.backend_url);

    const out = tauri.is_tauri ? await tauri_fetch(url, args) : await fetch(url, args);
    if (args != undefined && args.method != 'GET') {
        await notify_fellowship(data.character_id);
    }
    return out;
}

export async function get_role_dependend_backend(): Promise<BackendServer> {
    if (data.is_player) {
        return get_fellowship_backend();
    } else {
        return get_backend();
    }
}

async function notify_fellowship(character_id: number | null) {
    const socket = get_socket();
    if (socket != null) {
        await socket.send_message_to_socket(SocketMessage.UpdateFellowship, character_id);
    }
}
