import { extractComment } from './extractComment.js';
import { createIterator, } from './iterator.js';
import { generateReport } from './generateReport.js';
import { parseGeneral } from './tree/parseGeneral.js';
import { commentsMerger } from './tokenMergers/commentsMerger.js';
import { createMachine } from './mergerMachine.js';
import { stringMerger } from './tokenMergers/stringMerger.js';
import { templateStringMerger } from './tokenMergers/templateStringMerger.js';
import { closingTagMerger } from './tokenMergers/closingTagMerger.js';
import { typesAsMerger } from './tokenMergers/typesAsMerger.js';
import { typesCastMerger } from './tokenMergers/typesCastMerger.js';
import { customTCallMerger } from './tokenMergers/customTCallMerger.js';
export const DEFAULT_BLOCKS = {
    'block.begin': ['block.end'],
    'expression.begin': ['expression.end'],
    'expression.template.begin': ['expression.template.end'],
    'tag.regular.begin': [
        'tag.closing',
        'tag.self-closing.end',
        'tag.closing.begin',
    ],
};
export const DEFAULT_MERGERS = [
    stringMerger,
    templateStringMerger,
    closingTagMerger,
    typesAsMerger,
    typesCastMerger,
    customTCallMerger(['tolgee.t']),
];
export const Parser = ({ mappers, blocks, rules, merger, treeTransform, }) => {
    const ruleMap = new Map();
    rules.forEach(({ trigger, call }) => {
        ruleMap.set(trigger, call);
    });
    let iterator;
    function getCurrentLine() {
        return iterator.getLineNumber();
    }
    return {
        parse({ tokens, onAccept, options }) {
            var _a;
            for (const t of tokens) {
                // use first mapper, which gives some result
                const type = (_a = mappers.find((mapper) => mapper(t))) === null || _a === void 0 ? void 0 : _a(t);
                t.customType = type;
            }
            const mergedComments = [...createMachine(commentsMerger)(tokens)];
            const comments = mergedComments
                .filter((c) => c.customType === 'comment')
                .map((token) => extractComment(token))
                .filter(Boolean);
            // remove whitespaces and comments
            const filteredWhitespaces = mergedComments.filter((t) => {
                if (t.customType === 'comment') {
                    return false;
                }
                if (t.customType === 'ignore') {
                    return false;
                }
                if (!t.customType && t.token.match(/^[\s]+$/gmu)) {
                    return false;
                }
                return true;
            });
            const mergedTokens = [...merger(filteredWhitespaces)];
            // remove ignored after merge
            const filteredIgnored = mergedTokens.filter((t) => {
                if (t.customType === 'ignore') {
                    return false;
                }
                return true;
            });
            iterator = createIterator(filteredIgnored);
            const context = {
                tokens: iterator,
                getCurrentLine,
                withLabel,
                ruleMap,
                blocks,
            };
            let depth = 0;
            function withLabel(fn) {
                return (...args) => {
                    var _a;
                    let label;
                    const currentTokenName = ((_a = context.tokens.current()) === null || _a === void 0 ? void 0 : _a.customType) || '';
                    const isTrigger = currentTokenName.startsWith('trigger.');
                    if (!isTrigger) {
                        depth += 1;
                        label = `depth.${depth}`;
                    }
                    else {
                        label = currentTokenName;
                    }
                    const previous = iterator.getLabel();
                    iterator.setLabel(label);
                    const result = fn(...args);
                    iterator.setLabel(previous);
                    if (!isTrigger) {
                        depth -= 1;
                    }
                    return result;
                };
            }
            iterator.next();
            if (onAccept) {
                iterator.onAccept(onAccept);
            }
            let rootNode = parseGeneral(context, {
                end: [],
                keepNested: true,
            });
            const keys = [];
            const warnings = [];
            if (treeTransform) {
                const { tree, report } = treeTransform(rootNode);
                rootNode = tree;
                if (report) {
                    keys.push(...report.keys);
                    warnings.push(...report.warnings);
                }
            }
            const report = generateReport({ node: rootNode, comments, options });
            keys.push(...report.keys);
            warnings.push(...report.warnings);
            return {
                tree: rootNode,
                report: { keys, warnings },
            };
        },
    };
};
