import ansi from 'ansi-colors';
/**
 * Prints information about a key, with coloring and formatting.
 *
 * @param key The key to print.
 * @param deletion True if the key is about to be deleted.
 */
export function printKey(key, deletion, color, note) {
    const colorFunc = color !== null && color !== void 0 ? color : (deletion ? ansi.red : ansi.green);
    const namespace = key.namespace
        ? ` ${ansi.italic(`(namespace: ${key.namespace})`)}`
        : '';
    const renderedNote = note ? ` ${note}` : '';
    if (deletion === undefined) {
        console.log(`${colorFunc(`${key.keyName}`)}${namespace}${renderedNote}`);
    }
    else if (deletion) {
        console.log(`${colorFunc(`- ${key.keyName}`)}${namespace}${renderedNote}`);
    }
    else {
        console.log(`${colorFunc(`+ ${key.keyName}`)}${namespace}${renderedNote}`);
    }
}
/**
 * Compares local and remote keys to detect added and deleted keys.
 * **Warning**: `local` will be modified as a side-effect!
 *
 * @param local Local keys.
 * @param remote Remote keys.
 * @returns A list of added and removed keys.
 */
export function compareKeys(local, remote) {
    var _a;
    const result = { added: [], removed: [] };
    // Deleted keys
    for (const remoteKey of remote) {
        const namespace = remoteKey.namespace || '';
        const keyExists = (_a = local[namespace]) === null || _a === void 0 ? void 0 : _a.delete(remoteKey.name);
        if (!keyExists) {
            result.removed.push({
                id: remoteKey.id,
                keyName: remoteKey.name,
                namespace: remoteKey.namespace || '',
            });
        }
    }
    // Added keys
    const namespaces = [...Object.keys(local).sort()];
    for (const namespace of namespaces) {
        if (namespace in local && local[namespace].size) {
            const keys = local[namespace];
            const keyNames = Array.from(local[namespace].keys()).sort();
            for (const keyName of keyNames) {
                result.added.push({
                    keyName: keyName,
                    namespace: namespace || '',
                    defaultValue: keys.get(keyName) || undefined,
                });
            }
        }
    }
    // Sort keys
    // This is only necessary for unused keys, because the added keys are sorted directly as they're added.
    result.removed.sort((a, b) => {
        if (a.namespace === b.namespace) {
            return a.keyName > b.keyName ? 1 : a.keyName < b.keyName ? -1 : 0;
        }
        if (!a.namespace && b.namespace)
            return -1;
        if (a.namespace && !b.namespace)
            return 1;
        return a.namespace > b.namespace
            ? 1
            : a.namespace < b.namespace
                ? -1
                : 0;
    });
    return result;
}
