use axum::extract::{Path, State};
use axum::routing::{get, post, put};
use axum::{Json, Router};
use enums::calling::CallingEnum;
use enums::combat_skill::CombatSkillEnum;
use enums::culture::CultureEnum;
use enums::skill::SkillEnum;
use strum::EnumCount as _;

use crate::character::Character;
use crate::character::combat_skills::CombatSkills;
use crate::error::Result;
use crate::new_character::previous_experience::PreviousExperience;
use crate::new_character::{calling::Calling, choices::Choices, culture::Culture};
use crate::server::backend::AppState;
use crate::server::cache_response::CacheResponse;
use crate::server::users::User;

pub fn get_router() -> Router<AppState> {
    Router::new()
        .route("/empty/{strider}", get(get_choices))
        .route("/create", put(create_character))
        .route("/culture/{culture}", get(get_culture))
        .route("/calling/{calling}", get(get_calling))
        .route(
            "/skills/can-upgrade/{previous_experience}",
            post(get_can_upgrade_skill),
        )
        .route(
            "/combat-skills/can-upgrade/{previous_experience}",
            post(get_can_upgrade_combat_skill),
        )
        .route("/skill-cost/{new_value}", get(get_skill_cost))
        .route("/combat-skill-cost/{new_value}", get(get_combat_skill_cost))
        .route("/combat-skills", post(get_combat_skills))
}

async fn get_choices(Path(strider): Path<bool>) -> CacheResponse<Choices> {
    CacheResponse::new(Choices::new(strider))
}

async fn create_character(
    state: State<AppState>,
    user: User,
    Json(choices): Json<Choices>,
) -> Result<Json<Character>> {
    let mut character = Character::new(&choices, user.get_username().clone())?;
    let id = character.save_to_db(&state.db, &user).await?;
    character.set_id(id);
    Ok(Json(character))
}

async fn get_culture(Path(culture): Path<CultureEnum>) -> CacheResponse<Culture> {
    CacheResponse::new(Culture::new(&culture))
}

async fn get_calling(Path(calling): Path<CallingEnum>) -> CacheResponse<Calling> {
    CacheResponse::new(Calling::new(&calling))
}

async fn get_can_upgrade_skill(
    Path(previous_experience): Path<u32>,
    Json(skills): Json<[u32; SkillEnum::COUNT]>,
) -> Json<Vec<SkillEnum>> {
    let skills = PreviousExperience::get_can_upgrade_skills(previous_experience, &skills);
    Json(skills)
}

async fn get_can_upgrade_combat_skill(
    Path(previous_experience): Path<u32>,
    Json(skills): Json<[u32; CombatSkillEnum::COUNT]>,
) -> Json<Vec<CombatSkillEnum>> {
    let skills = PreviousExperience::get_can_upgrade_combat_skills(previous_experience, &skills);
    Json(skills)
}

async fn get_skill_cost(Path(new_value): Path<u32>) -> CacheResponse<u32> {
    let cost = PreviousExperience::get_skill_cost(new_value);
    CacheResponse::new(cost)
}

async fn get_combat_skill_cost(Path(new_value): Path<u32>) -> CacheResponse<u32> {
    let cost = PreviousExperience::get_combat_skill_cost(new_value);
    CacheResponse::new(cost)
}

async fn get_combat_skills(Json(choices): Json<Choices>) -> Result<Json<CombatSkills>> {
    let skills = CombatSkills::new(&choices, false)?;
    Ok(Json(skills))
}
