use axum::extract::{Path, Query, State};
use axum::routing::{delete, get, post, put};
use axum::{Json, Router};
use enums::combat_skill::CombatSkillEnum;
use enums::skill::SkillEnum;
use serde::Deserialize;

use crate::character::reward::Reward;
use crate::character::roll::CharacterRoll;
use crate::character::skill_type::SkillType;
use crate::character::virtue::Virtue;
use crate::character::{Character, secondary_attributes::SecondaryAttributeEnum};
use crate::error::Result;
use crate::fellowship::undertaking::UndertakingEnum;
use crate::roll::RollType;
use crate::server::backend::AppState;
use crate::server::users::User;

pub fn get_router() -> Router<AppState> {
    axum::Router::new()
        .route("/{character_id}", get(get_character))
        .route("/{character_id}", delete(delete_character))
        .route("/", get(get_list_character))
        .route("/", put(set_character))
        .route(
            "/{character_id}/loose-endurance/{value}",
            post(loose_endurance),
        )
        .route("/{character_id}/loose-hope/{value}", post(loose_hope))
        .route(
            "/{character_id}/skills/can-upgrade",
            get(skills_can_upgrade),
        )
        .route(
            "/{character_id}/combat-skills/can-upgrade",
            get(combat_skills_can_upgrade),
        )
        .route(
            "/{character_id}/secondary-attributes/can-upgrade",
            get(secondary_attributes_can_upgrade),
        )
        .route(
            "/{character_id}/increase-skill/{skill}/{delta}",
            post(increase_skill),
        )
        .route(
            "/{character_id}/increase-combat-skill/{skill}/{delta}",
            post(increase_combat_skill),
        )
        .route(
            "/{character_id}/increase-secondary-attribute/Wisdom/{delta}",
            post(increase_wisdom),
        )
        .route(
            "/{character_id}/increase-secondary-attribute/Valor/{delta}",
            post(increase_valor),
        )
        .route("/{character_id}/notes", post(update_notes))
        .route("/{character_id}/notes", get(get_notes))
        .route("/{character_id}/Valor/advantage", post(add_virtue))
        .route("/{character_id}/Wisdom/advantage", post(add_reward))
        .route("/{character_id}/long-rest", post(long_rest))
        .route("/{character_id}/short-rest", post(short_rest))
        .route(
            "/{character_id}/fellowship-phase/{undertaking}/{yule}",
            post(take_fellowship_phase),
        )
        .route(
            "/{character_id}/reward-xp/skill/{skill}/adventure/{adventure}",
            post(reward_xp),
        )
        .route("/{character_id}/convert-shadow", post(convert_shadow))
        .route("/{character_id}/roll", post(roll_skill))
}

async fn get_character(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<Json<Character>> {
    let character = Character::get(&state.db, &user, character_id).await?;
    Ok(Json(character))
}

async fn delete_character(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<()> {
    Character::delete_from_db(&state.db, character_id, &user).await?;
    Ok(())
}

async fn get_list_character(state: State<AppState>, user: User) -> Result<Json<Vec<i32>>> {
    let out = Character::get_list_characters(&state.db, &user).await?;
    Ok(Json(out))
}

async fn set_character(
    state: State<AppState>,
    user: User,
    Json(character): Json<Character>,
) -> Result<Json<i32>> {
    let id = character.save_to_db(&state.db, &user).await?;
    Ok(Json(id))
}

async fn loose_endurance(
    state: State<AppState>,
    user: User,
    Path((character_id, value)): Path<(i32, i32)>,
) -> Result<Json<u32>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;

    character.characteristics.loose_endurance(value);

    character.save_to_db(&state.db, &user).await?;

    Ok(Json(
        character.characteristics.attributes.endurance.get_current(),
    ))
}

async fn loose_hope(
    state: State<AppState>,
    user: User,
    Path((character_id, value)): Path<(i32, i32)>,
) -> Result<Json<u32>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;

    character.characteristics.loose_hope(value);

    character.save_to_db(&state.db, &user).await?;

    Ok(Json(
        character.characteristics.attributes.hope.get_current(),
    ))
}

async fn skills_can_upgrade(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<Json<Vec<SkillEnum>>> {
    let character = Character::get(&state.db, &user, character_id).await?;
    Ok(Json(character.characteristics.can_upgrade_skills()))
}

async fn combat_skills_can_upgrade(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<Json<Vec<CombatSkillEnum>>> {
    let character = Character::get(&state.db, &user, character_id).await?;
    Ok(Json(character.characteristics.can_upgrade_combat_skills()))
}

async fn secondary_attributes_can_upgrade(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<Json<Vec<SecondaryAttributeEnum>>> {
    let character = Character::get(&state.db, &user, character_id).await?;
    Ok(Json(
        character.characteristics.can_upgrade_secondary_attributes(),
    ))
}

#[derive(Deserialize)]
struct OptionalBool {
    value: Option<bool>,
}

async fn increase_skill(
    state: State<AppState>,
    user: User,
    Path((character_id, skill, delta)): Path<(i32, SkillEnum, i32)>,
    Query(OptionalBool { value: free }): Query<OptionalBool>,
) -> Result<Json<u32>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    let new_value =
        character
            .characteristics
            .increase_skill(&skill, delta, free.unwrap_or(false))?;
    character.save_to_db(&state.db, &user).await?;
    Ok(Json(new_value))
}

async fn increase_combat_skill(
    state: State<AppState>,
    user: User,
    Path((character_id, skill, delta)): Path<(i32, CombatSkillEnum, i32)>,
    Query(OptionalBool { value: free }): Query<OptionalBool>,
) -> Result<Json<u32>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    let new_value =
        character
            .characteristics
            .increase_combat_skill(&skill, delta, free.unwrap_or(false))?;
    character.save_to_db(&state.db, &user).await?;
    Ok(Json(new_value))
}

async fn increase_wisdom(
    state: State<AppState>,
    user: User,
    Path((character_id, delta)): Path<(i32, i32)>,
    Query(OptionalBool { value: free }): Query<OptionalBool>,
    Json(virtue): Json<Virtue>,
) -> Result<Json<(u32, Vec<Virtue>)>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    let new_value =
        character
            .characteristics
            .increase_wisdom(&virtue, delta, free.unwrap_or(false))?;
    character.save_to_db(&state.db, &user).await?;

    // Update ids
    let character = Character::get(&state.db, &user, character_id).await?;
    let new_virtues = character.characteristics.get_virtues();
    Ok(Json((new_value, new_virtues.to_vec())))
}

async fn increase_valor(
    state: State<AppState>,
    user: User,
    Path((character_id, delta)): Path<(i32, i32)>,
    Query(OptionalBool { value: free }): Query<OptionalBool>,
    Json(reward): Json<Reward>,
) -> Result<Json<(u32, Vec<Reward>)>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    let new_value =
        character
            .characteristics
            .increase_valor(&reward, delta, free.unwrap_or(false))?;
    character.save_to_db(&state.db, &user).await?;

    // Update ids
    let character = Character::get(&state.db, &user, character_id).await?;
    let new_rewards = character.characteristics.get_rewards();
    Ok(Json((new_value, new_rewards.to_vec())))
}

async fn update_notes(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
    Json(notes): Json<String>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.set_notes(&notes);
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn get_notes(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<Json<String>> {
    let character = Character::get(&state.db, &user, character_id).await?;
    Ok(Json(character.get_notes().clone()))
}

async fn add_reward(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
    Json(adv): Json<Reward>,
) -> Result<Json<Vec<Reward>>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.characteristics.valor.add_advantage(&adv);
    character.save_to_db(&state.db, &user).await?;
    let ret = character.characteristics.get_rewards();
    Ok(Json(ret.to_vec()))
}

async fn add_virtue(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
    Json(adv): Json<Virtue>,
) -> Result<Json<Vec<Virtue>>> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.characteristics.wisdom.add_advantage(&adv);
    character.save_to_db(&state.db, &user).await?;
    let ret = character.characteristics.get_virtues();
    Ok(Json(ret.to_vec()))
}

async fn long_rest(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.rest(true);
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn short_rest(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.rest(false);
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn take_fellowship_phase(
    state: State<AppState>,
    user: User,
    Path((character_id, undertaking, yule)): Path<(i32, UndertakingEnum, bool)>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.take_fellowship_phase(yule, undertaking)?;
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn reward_xp(
    state: State<AppState>,
    user: User,
    Path((character_id, skill, adventure)): Path<(i32, i32, i32)>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.characteristics.reward_xp(skill, adventure)?;
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn convert_shadow(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.characteristics.attributes.shadow.convert();
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

#[derive(Deserialize)]
struct RollSkillOptions {
    bonus_dices: i32,
    roll_type: Option<RollType>,
}

async fn roll_skill(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
    Query(RollSkillOptions {
        bonus_dices,
        roll_type,
    }): Query<RollSkillOptions>,
    Json(skill): Json<SkillType>,
) -> Result<Json<CharacterRoll>> {
    let character = Character::get(&state.db, &user, character_id).await?;
    Ok(Json(character.roll(&skill, bonus_dices, roll_type)))
}
