use anyhow::{Result, anyhow};
use axum::extract::ws::{Message, WebSocket};
use serde::{Deserialize, Serialize};
use std::sync::LazyLock;
use std::{collections::HashMap, sync::Arc};
use tokio::sync::Mutex;
use tracing::info;
use ts_rs::TS;

use super::routes::socket::Event;

// Define a few types
pub type WebSocketMessage = Message;
pub type SafeSocket = Arc<Mutex<WebSocket>>;
pub type Conn = Arc<Mutex<HashMap<String, ConnectionHandler>>>;

// Define the global connection handler
pub static CONNECTIONS: LazyLock<Conn> = LazyLock::new(|| Arc::new(Mutex::new(HashMap::new())));

#[derive(Serialize, Deserialize, Clone, TS)]
#[ts(export)]
pub struct Connection {
    pub player: String,
    pub loremaster: String,
    pub fellowship: String,
}

pub struct ConnectionHandler {
    connections: HashMap<String, SafeSocket>,
}

impl ConnectionHandler {
    pub fn new(name: String, socket: SafeSocket) -> Self {
        let mut out = Self {
            connections: HashMap::new(),
        };
        out.add(name, socket);
        out
    }

    pub async fn update_everyone(loremaster: String, event: Event) {
        let mut handler = CONNECTIONS.lock().await;
        let conn = handler.get_mut(&loremaster);
        info!("updating {loremaster}");
        if let Some(conn) = conn {
            let err = conn.send_to_players(event).await;
            if let Err(x) = err {
                eprintln!("Error while updating everyone: {x:?}");
            }
        }
    }

    pub async fn update_single(loremaster: String, player: &str, event: Event) {
        let mut handler = CONNECTIONS.lock().await;
        let conn = handler.get_mut(&loremaster);
        info!("updating message {loremaster} - {player}");
        if let Some(conn) = conn {
            let err = conn.send_to_single_player(player, event).await;
            if let Err(x) = err {
                eprintln!("Error while updating everyone: {x:?}");
            }
        }
    }

    pub fn add(&mut self, conn: String, socket: SafeSocket) {
        info!("Adding a connection {conn}");
        self.connections.insert(conn, socket);
    }

    pub async fn remove(&mut self, user: &str) {
        info!("Removing a connection {user}");
        self.connections.remove(user);
        self.remove_terminated_connections().await;
    }

    pub async fn send_to_single_player<T>(&mut self, username: &str, obj: T) -> Result<()>
    where
        T: Serialize,
    {
        let text = serde_json::to_string(&obj)?;
        let conn = self
            .connections
            .get(username)
            .ok_or(anyhow!(format!("Cannot find user {username}")))?;
        let mut conn = conn.lock().await;
        conn.send(WebSocketMessage::Text(text.into())).await?;
        Ok(())
    }

    async fn send_to_players<T>(&mut self, obj: T) -> Result<()>
    where
        T: Serialize,
    {
        let text = serde_json::to_string(&obj)?;
        let mut status = Ok(());
        for locked_conn in &self.connections {
            let mut conn = locked_conn.1.lock().await;
            match conn.send(WebSocketMessage::Text(text.clone().into())).await {
                Ok(()) => {}
                Err(x) => status = Err(x),
            }
        }
        status?;
        Ok(())
    }

    pub async fn cleanup() {
        info!("Cleanup sockets");
        let mut handler = CONNECTIONS.lock().await;
        handler.retain(|_, x| !x.connections.is_empty());
    }

    async fn remove_terminated_connections(&mut self) {
        let mut keys: Vec<String> = vec![];
        for locked_conn in &self.connections {
            let conn = locked_conn.1.lock().await;
            if conn.is_terminated() {
                keys.push(locked_conn.0.clone());
            }
        }
        for key in &keys {
            self.connections.remove(key);
        }
    }
}
