use serde::{Deserialize, Serialize};
use std::{
    fs::File,
    path::{Path, PathBuf},
    sync::LazyLock,
};
use tokio::sync::Mutex;

pub static CONFIG: LazyLock<Mutex<Config>> = LazyLock::new(|| Mutex::new(Config::default()));

const CONFIG_FILE: &str = "config.json";

#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct Config {
    #[serde(alias = "expose_rocket")]
    expose_server: bool,
    frontend_path: String,
    #[serde(default)]
    directory: PathBuf,
    #[serde(default)]
    configured: bool,
}

impl Default for Config {
    fn default() -> Self {
        Self {
            frontend_path: String::from("frontend/build"),
            expose_server: false,
            directory: PathBuf::new(),
            configured: false,
        }
    }
}

impl Config {
    pub fn configure(&mut self, config_dir: &Path) {
        eprintln!("config: {}", config_dir.display());
        self.directory = config_dir.to_path_buf();
        self.configured = true;

        let config_file = config_dir.join(CONFIG_FILE);
        let Ok(file) = File::open(config_file) else {
            return;
        };
        let config = match serde_json::from_reader::<File, Config>(file) {
            Err(error) => {
                eprintln!("Failed to read config file: {error}");
                self.clone()
            }
            Ok(mut config) => {
                eprintln!("Successfully read configuration file");
                if cfg!(target_os = "android") {
                    config.directory = config_dir.to_path_buf();
                }
                config
            }
        };
        self.clone_from(&config);
        self.configured = true;
    }

    pub fn is_configured(&self) -> bool {
        self.configured
    }

    pub fn expose_server(&self) -> bool {
        self.expose_server
    }

    pub fn frontend_path(&self) -> &String {
        &self.frontend_path
    }

    pub fn get_directory(&self) -> &PathBuf {
        &self.directory
    }
}
