use enums::character_data::CharacterData;
use enums::combat_proficiency::CombatProficiency;
use enums::weapon::WeaponEnum;
use sea_orm::ActiveValue::{NotSet, Set};
use serde::{Deserialize, Serialize};
use ts_rs::TS;

use crate::assets::read_enum_from_assets;
use crate::character::reward::Reward;
use crate::common::computed_value::ComputedValue;
use crate::common::enums::find_enum;
use crate::entity::reward_weapon;
use crate::entity::weapon as entity;
use crate::impl_item_trait;
use crate::utils::convert_option_to_set;

const DIRECTORY: &str = "items/weapons";

#[derive(Serialize, Deserialize, Clone, Default, TS, Debug)]
#[ts(export)]
pub struct Weapon {
    id: Option<i32>,
    name: String,
    enum_value: Option<WeaponEnum>,
    #[serde(alias = "one_hand_damage")]
    damage: ComputedValue<u32>,
    injury: ComputedValue<u32>,
    two_hand_injury: Option<ComputedValue<u32>>,
    load: ComputedValue<u32>,
    #[serde(default)]
    combat_proficiency: CombatProficiency,
    #[serde(default)]
    wearing: bool,
    #[serde(default)]
    notes: String,
    #[serde(default)]
    famous: bool,
    #[serde(default)]
    rewards: Vec<Reward>,
}

impl_item_trait!(
    Weapon,
    reward_weapon::Entity,
    entity::Entity,
    weapon_id,
    reward_weapon::ActiveModel
);

impl Weapon {
    pub fn new(
        name: String,
        damage: u32,
        injury: u32,
        two_hand_injury: Option<u32>,
        load: u32,
        combat_proficiency: CombatProficiency,
    ) -> Self {
        Self {
            id: None,
            name,
            enum_value: None,
            damage: ComputedValue::new(damage),
            injury: ComputedValue::new(injury),
            two_hand_injury: two_hand_injury.map(ComputedValue::new),
            load: ComputedValue::new(load),
            combat_proficiency,
            wearing: true,
            notes: String::new(),
            famous: false,
            rewards: vec![],
        }
    }

    #[deprecated(since = "3.0.0", note = "Please use the sql functions")]
    pub fn reset_id(&mut self) {
        self.id = None;
    }
    fn compute_special_derived_values(&mut self, data: &CharacterData) {
        let injury_diff = self.rewards.iter().fold(0, |acc, rew| {
            acc + if let Some(reward) = rew.get_enum() {
                reward.increase_injury(data)
            } else {
                0
            }
        });
        self.injury
            .set_computed_value(self.injury.value + injury_diff);
        if let Some(mut injury) = self.two_hand_injury {
            injury.set_computed_value(injury.value + injury_diff);
        }

        let damage_diff = self.rewards.iter().fold(0, |acc, rew| {
            acc + if let Some(reward) = rew.get_enum() {
                reward.increase_damage(
                    data,
                    self.combat_proficiency.get_skill_value(data.combat_skills),
                )
            } else {
                0
            }
        });
        self.damage
            .set_computed_value(self.damage.value + damage_diff);
    }
}

impl From<WeaponEnum> for Weapon {
    fn from(value: WeaponEnum) -> Self {
        let mut out: Self = read_enum_from_assets(&value, DIRECTORY);
        out.enum_value = Some(value);
        out
    }
}

impl From<&str> for Weapon {
    fn from(value: &str) -> Self {
        let value = format!("\"{value}\"");
        let value: WeaponEnum =
            serde_json::from_str(&value).expect("Failed to convert to weapon enum");
        Self::from(value)
    }
}

impl From<entity::Model> for Weapon {
    fn from(value: entity::Model) -> Self {
        let prof = serde_json::from_str(&value.combat_proficiency)
            .expect("SQL and serde enums are not compatible");
        let enum_value = value.enum_value.as_ref().and_then(|name| {
            serde_json::from_str(name).expect("SQL and serde enums are not compatible")
        });

        Self {
            id: Some(value.id),
            enum_value: match enum_value {
                Some(x) => Some(x),
                None => find_enum(&value.name),
            },
            name: value.name,
            damage: ComputedValue::new(value.damage.try_into().expect("Failed to convert")),
            injury: ComputedValue::new(value.injury.try_into().expect("Failed to convert")),
            two_hand_injury: value
                .two_hand_injury
                .map(|x| ComputedValue::new(x.try_into().expect("Failed to convert"))),
            load: ComputedValue::new(value.load.try_into().expect("Failed to convert")),
            combat_proficiency: prof,
            wearing: value.wearing,
            notes: value.notes,
            famous: value.famous,
            rewards: vec![],
        }
    }
}

impl From<Weapon> for entity::ActiveModel {
    fn from(value: Weapon) -> Self {
        let id = convert_option_to_set(value.id);
        let prof = serde_json::to_string(&value.combat_proficiency)
            .expect("Failed to convert enum to string");
        let enum_value = value
            .enum_value
            .as_ref()
            .map(|value| serde_json::to_string(value).expect("Failed to convert enum to string"));
        Self {
            id,
            name: Set(value.name),
            enum_value: Set(enum_value),
            damage: Set(value.damage.value.try_into().expect("Failed to convert")),
            injury: Set(value.injury.value.try_into().expect("Failed to convert")),
            two_hand_injury: Set(value
                .two_hand_injury
                .map(|x| x.value.try_into().expect("Failed to convert"))),
            load: Set(value.load.value.try_into().expect("Failed to convert")),
            combat_proficiency: Set(prof),
            wearing: Set(value.wearing),
            notes: Set(value.notes),
            character_id: NotSet,
            famous: Set(value.famous),
        }
    }
}

#[cfg(test)]
mod test {
    use strum::{EnumCount as _, IntoEnumIterator as _};

    use crate::assets::get_number_files;

    use super::*;

    #[test]
    fn check_weapon_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(WeaponEnum::COUNT == number_files);

        for weapon in WeaponEnum::iter() {
            eprintln!("Testing {weapon:?}");
            let _: Weapon = Weapon::from(weapon);
        }
    }
}
