use crate::entity::fell_ability as entity;
use anyhow::Result;
use enums::fell_ability::FellAbilityEnum;
use sea_orm::ActiveValue::NotSet;
use sea_orm::{
    ActiveModelTrait as _, ColumnTrait as _, ConnectionTrait, DatabaseTransaction,
    EntityTrait as _, QueryFilter as _, Set,
};
use serde::de::Error as SerdeError;
use serde::{Deserialize, Deserializer, Serialize};
use serde_json::value::RawValue;
use ts_rs::TS;

use crate::assets::read_enum_from_assets;
use crate::utils::convert_option_to_set;

const DIRECTORY: &str = "adversaries/fell_abilities";

#[derive(Serialize, Deserialize, TS, Clone, Debug)]
#[ts(export)]
pub struct FellAbility {
    pub id: Option<i32>,
    pub enum_value: Option<FellAbilityEnum>,
    pub title: String,
    pub description: String,
}

impl From<FellAbilityEnum> for FellAbility {
    fn from(item: FellAbilityEnum) -> Self {
        let mut out: Self = read_enum_from_assets(&item, DIRECTORY);
        out.enum_value = Some(item);
        out
    }
}

// For enum and full ability
pub fn deserialize_fell_abilities<'de, D>(deserializer: D) -> Result<Vec<FellAbility>, D::Error>
where
    D: Deserializer<'de>,
{
    let json: Vec<Box<RawValue>> = Deserialize::deserialize(deserializer)?;
    let mut out = vec![];
    for ability in &json {
        let ability = match serde_json::from_str::<FellAbilityEnum>(ability.get()) {
            Ok(x) => FellAbility::from(x),
            Err(_) => {
                serde_json::from_str::<FellAbility>(ability.get()).map_err(SerdeError::custom)?
            }
        };
        out.push(ability);
    }
    Ok(out)
}

impl From<entity::Model> for FellAbility {
    fn from(value: entity::Model) -> Self {
        let enum_value = value.enum_value.as_ref().and_then(|name| {
            serde_json::from_str(name).expect("SQL and serde enums are not compatible")
        });

        Self {
            id: Some(value.id),
            enum_value,
            title: value.title,
            description: value.description,
        }
    }
}

impl From<FellAbility> for entity::ActiveModel {
    fn from(value: FellAbility) -> Self {
        let id = convert_option_to_set(value.id);
        let enum_value = value
            .enum_value
            .as_ref()
            .map(|value| serde_json::to_string(value).expect("Failed to convert enum to string"));

        Self {
            id,
            title: Set(value.title),
            enum_value: Set(enum_value),
            description: Set(value.description),
            adversary_id: NotSet,
        }
    }
}

impl FellAbility {
    pub async fn save(&self, db: &DatabaseTransaction, adversary_id: i32) -> Result<()> {
        let mut model: entity::ActiveModel = (*self).clone().into();
        model.adversary_id = Set(adversary_id);
        model.save(db).await?;
        Ok(())
    }

    pub async fn delete_by_adversary_id<C: ConnectionTrait>(
        db: &C,
        adversary_id: i32,
    ) -> Result<()> {
        entity::Entity::delete_many()
            .filter(entity::Column::AdversaryId.eq(adversary_id))
            .exec(db)
            .await?;
        Ok(())
    }

    pub fn reset_hate_on_creation(&self) -> bool {
        self.enum_value != Some(FellAbilityEnum::UnnaturalHunger)
    }
}

#[cfg(test)]
mod test {
    use strum::{EnumCount as _, IntoEnumIterator as _};

    use crate::{
        adversary::fell_ability::{DIRECTORY, FellAbility},
        assets::get_number_files,
    };

    use super::*;

    #[test]
    fn reset_hate_on_creation() {
        let value: FellAbility =
            read_enum_from_assets(&FellAbilityEnum::UnnaturalHunger, DIRECTORY);
        assert!(value.reset_hate_on_creation());
    }

    #[test]
    fn check_fell_ability_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(FellAbilityEnum::COUNT == number_files);

        for ability in FellAbilityEnum::iter() {
            eprintln!("Testing {ability:?}");
            let _: FellAbility = FellAbility::from(ability);
        }
    }
}
