use crate::{
    assets::read_enum_from_assets, entity::adversary_combat_proficiency as entity,
    utils::convert_option_to_set,
};
use anyhow::Result;
use sea_orm::{
    ActiveModelTrait as _, ActiveValue::NotSet, ColumnTrait as _, ConnectionTrait,
    DatabaseTransaction, EntityTrait as _, QueryFilter as _, Set,
};
use serde::de::Error as SerdeError;
use serde::{Deserialize, Deserializer, Serialize};
use serde_json::value::RawValue;
use strum::{EnumCount, EnumIter};
use ts_rs::TS;

const DIRECTORY: &str = "adversaries/weapons";

#[derive(Serialize, Deserialize, EnumIter, EnumCount, Debug, Clone, Copy)]
enum WeaponEnum {
    AncientSword,
    Axe,
    BattleAxe,
    Bow,
    BowOfHorn,
    BroadBladedSword,
    BroadHeadedSpear,
    Club,
    Crush,
    Cudgel,
    GreatBow,
    GreatSpear,
    HeavyScimitar,
    JaggedKnife,
    LongHaftedAxe,
    LongSword,
    OrcAxe,
    Scimitar,
    ShortSpear,
    ShortSword,
    Spear,
    Staff,
    Sword,
    Tentacle,
    TorchStaff,
}
#[derive(Debug, Deserialize)]
struct Weapon {
    damage: u32,
    injury: u32,
    special_damages: Vec<String>,
    #[expect(clippy::struct_field_names)]
    weapon: String,
}

impl Weapon {
    fn new(weapon: WeaponEnum) -> Self {
        read_enum_from_assets(&weapon, DIRECTORY)
    }
}

#[derive(Serialize, Deserialize, TS, Clone, Debug)]
#[ts(export)]
pub struct AdversaryCombatProficiency {
    id: Option<i32>,
    weapon: String,
    rating: u32,
    damage: u32,
    injury: u32,
    special_damages: Vec<String>,
}

pub fn deserialize_weapon<'de, D>(
    deserializer: D,
) -> Result<Vec<AdversaryCombatProficiency>, D::Error>
where
    D: Deserializer<'de>,
{
    #[derive(Debug, Deserialize)]
    struct ProficiencyWithWeapon {
        name: WeaponEnum,
        rating: u32,
    }

    let json: Vec<Box<RawValue>> = Deserialize::deserialize(deserializer)?;
    let mut out = vec![];
    for prof in &json {
        if let Ok(x) = serde_json::from_str::<AdversaryCombatProficiency>(prof.get()) {
            out.push(x);
            continue;
        }
        let x = serde_json::from_str::<ProficiencyWithWeapon>(prof.get())
            .map_err(|error| SerdeError::custom(error.to_string()))?;
        let weapon = Weapon::new(x.name);
        out.push(AdversaryCombatProficiency {
            id: None,
            weapon: weapon.weapon,
            rating: x.rating,
            damage: weapon.damage,
            injury: weapon.injury,
            special_damages: weapon.special_damages,
        });
    }
    Ok(out)
}

impl From<entity::Model> for AdversaryCombatProficiency {
    fn from(value: entity::Model) -> Self {
        Self {
            id: Some(value.id),
            weapon: value.weapon,
            rating: value.rating.try_into().expect("Failed to convert"),
            damage: value.damage.try_into().expect("Failed to convert"),
            injury: value.injury.try_into().expect("Failed to convert"),
            special_damages: value.special_damages.split(',').map(String::from).collect(),
        }
    }
}

impl From<AdversaryCombatProficiency> for entity::ActiveModel {
    fn from(value: AdversaryCombatProficiency) -> Self {
        let id = convert_option_to_set(value.id);
        Self {
            id,
            weapon: Set(value.weapon),
            rating: Set(value.rating.try_into().expect("Failed to convert")),
            damage: Set(value.damage.try_into().expect("Failed to convert")),
            injury: Set(value.injury.try_into().expect("Failed to convert")),
            special_damages: Set(value.special_damages.join(",")),
            adversary_id: NotSet,
        }
    }
}

impl AdversaryCombatProficiency {
    pub async fn save(&self, db: &DatabaseTransaction, adversary_id: i32) -> Result<()> {
        let mut model: entity::ActiveModel = (*self).clone().into();
        model.adversary_id = Set(adversary_id);
        model.save(db).await?;
        Ok(())
    }
    pub async fn delete_by_adversary_id<C: ConnectionTrait>(
        db: &C,
        adversary_id: i32,
    ) -> Result<()> {
        entity::Entity::delete_many()
            .filter(entity::Column::AdversaryId.eq(adversary_id))
            .exec(db)
            .await?;
        Ok(())
    }
}

#[cfg(test)]
mod test {
    use crate::assets::get_number_files;

    use super::*;
    use strum::IntoEnumIterator as _;

    #[test]
    fn check_adversaries_weapon_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(WeaponEnum::COUNT == number_files);

        for weapon in WeaponEnum::iter() {
            eprintln!("Testing {weapon:?}");
            let _: Weapon = Weapon::new(weapon);
        }
    }
}
