import type { RequestOrigin } from './bindings/RequestOrigin';
import { Err, MyError, Ok, type Try } from '../utils/error';
import { sleep } from '../utils/utils';
import { BackendServer } from './queries';

export type FellowshipAuth = {
    fellowship: string;
    loremaster: string;
    player: string;
    server: URL;
};

export function get_origin(url: URL): RequestOrigin {
    return url.host == window.location.host ? 'Local' : 'Extern';
}

export enum SocketMessage {
    Version = 'Version',
    UpdateFellowship = 'UpdateFellowship',
    UpdateMessage = 'UpdateMessage'
}

export class Socket {
    socket: null | WebSocket = null;
    // Backend of the player not the socket
    backend: BackendServer;
    fellowship_auth: FellowshipAuth;
    process_events: (event: MessageEvent) => Promise<void>;

    constructor(
        backend: BackendServer,
        fellowship_auth: FellowshipAuth,
        process_events: (event: MessageEvent) => Promise<void>
    ) {
        this.backend = backend;
        this.socket = null;
        this.process_events = process_events;
        this.fellowship_auth = fellowship_auth;
    }

    set(s: WebSocket) {
        this.socket = s;
    }

    is_set(): boolean {
        return this.socket != null;
    }

    send(data: string) {
        if (this.socket == null) {
            throw 'logic error';
        }
        this.socket.send(data);
    }

    async send_message_to_socket(
        message: SocketMessage,
        character_id: number | null
    ): Promise<Try<void>> {
        if (!this.is_set()) {
            return Ok(undefined);
        }
        if (message == SocketMessage.UpdateFellowship) {
            const auth = this.fellowship_auth;
            if (get_origin(auth.server) == 'Extern') {
                if (character_id == null) {
                    return Err(new MyError('No character id found, cannot update fellowship'));
                }
                const c = (await this.backend.get_character(character_id)).expect(
                    'Failed to get character'
                );
                this.send(JSON.stringify({ Character: c }));
            }
        }
        this.send(JSON.stringify(message));
        return Ok(undefined);
    }

    // Setup the socket and enable its connection for players
    // if credentials are in cache
    async setup_socket(is_player: boolean) {
        let server = null;
        if (is_player) {
            const auth = this.fellowship_auth;
            server = auth.server;
        } else {
            server = this.backend.backend_url;
        }
        server = server.toString().replace('http', 'ws');
        const ws = new WebSocket(server + 'socket/fellowship');

        // Authenticate request
        const wait = 100;
        while (this.backend.auth == null) {
            await sleep(wait);
        }
        ws.addEventListener('open', async () => {
            const version = (await this.backend.get_version()).expect('Failed to get the version');
            // Check version
            ws.send(JSON.stringify(version));
            if (is_player) {
                const fellowship = this.fellowship_auth;
                const auth = {
                    player: fellowship.player,
                    loremaster: fellowship.loremaster,
                    fellowship: fellowship.fellowship
                };
                ws.send(JSON.stringify({ Player: auth }));
            } else {
                ws.send(JSON.stringify({ Loremaster: this.backend.auth }));
            }
            console.log('Socket connection has been established');
        });
        this.set(ws);

        // Listen to incoming messages
        ws.addEventListener('message', this.process_events);
    }
}
