import clsx from 'clsx';
import React, { useState } from 'react';
import { styled, IconButton } from '@mui/material';
import { Clear } from '@mui/icons-material';

import { ScreenshotInterface } from '../dialogContext/useGallery';
import { ScreenshotWithLabels } from './ScreenshotWithLabels';
import { Tooltip } from '../../common/Tooltip';

const ScreenshotBox = styled('div')`
  border: 1px solid ${({ theme }) => theme.palette.grey[300]};
  position: relative;
  box-sizing: border-box;
  width: 98px;
  height: 98px;
  align-items: center;
  justify-content: center;
  display: flex;
  margin: 1px;
  cursor: pointer;
  overflow: visible;
`;

const ScreenshotOverflowWrapper = styled('div')`
  display: flex;
  align-items: center;
  justify-content: center;
  overflow: hidden;
  width: 100%;
  height: 100%;
`;

const DeleteIconButton = styled(IconButton)`
  position: absolute;
  z-index: 2;
  font-size: 20px;
  right: -8px;
  top: -8px;
  padding: 2px;
  background-color: rgba(62, 62, 62, 0.9);
  color: rgba(255, 255, 255, 0.8);
  visibility: hidden;
  opacity: 0;
  transition:
    visibility 0.1s linear,
    opacity 0.1s linear;
  &:hover {
    background-color: rgba(62, 62, 62, 1);
    color: rgba(255, 255, 255, 0.9);
  }
  &.hover {
    opacity: 1;
    visibility: visible;
  }
`;

const DeleteIcon = styled(Clear)`
  font-size: 20px;
`;

export type Props = {
  onClick: () => void;
  onDelete?: (id: number) => void;
  data: ScreenshotInterface;
  keyId: number | undefined;
};

export const ScreenshotThumbnail: React.FC<Props> = (props) => {
  const [hover, setHover] = useState(false);

  const onMouseOver = () => {
    setHover(true);
  };

  const onMouseOut = () => {
    setHover(false);
  };

  const onDeleteClick = () => {
    props.onDelete?.(props.data.id!);
  };

  const screenshot = props.data;

  return (
    <ScreenshotBox onMouseOver={onMouseOver} onMouseOut={onMouseOut}>
      {props.onDelete && (
        <Tooltip title="Delete">
          <DeleteIconButton onClick={onDeleteClick} className={clsx({ hover })}>
            <DeleteIcon />
          </DeleteIconButton>
        </Tooltip>
      )}

      <ScreenshotOverflowWrapper
        key={props.data.id}
        onClick={props.onClick}
        aria-label="Screenshot"
      >
        <ScreenshotWithLabels
          screenshot={{
            src: screenshot.fileUrl,
            width: screenshot.width,
            height: screenshot.height,
            highlightedKeyId: props.keyId ?? -1,
            keyReferences: screenshot.keyReferences,
          }}
        />
      </ScreenshotOverflowWrapper>
    </ScreenshotBox>
  );
};
