import { TolgeeOptions, TolgeePlugin, DevCredentials, TFnType, DefaultParamType, TranslationKey } from './types';
declare function createTolgee(options: TolgeeOptions): Readonly<{
    /**
     * Listen to tolgee events.
     */
    on: import("./types").TolgeeOn<keyof import("./types").EventType>;
    /**
     * Turn off/on events emitting. Is on by default.
     */
    setEmitterActive: (active: boolean) => void;
    /**
     * @return current language if set.
     */
    getLanguage: () => string | undefined;
    /**
     * `pendingLanguage` represents language which is currently being loaded.
     * @return current `pendingLanguage` if set.
     */
    getPendingLanguage: () => string | undefined;
    /**
     * Change current language.
     * - if not running sets `pendingLanguage`, `language` to the new value
     * - if running sets `pendingLanguage` to the value, fetches necessary data and then changes `language`
     *
     * @return Promise which is resolved when `language` is changed.
     */
    changeLanguage: (language: string) => Promise<void>;
    /**
     * Temporarily change translation in cache.
     * @return object with revert method.
     */
    changeTranslation: (descriptor: import("./types").CacheDescriptor, key: string, value: string) => {
        revert(): void;
    };
    /**
     * Adds namespace(s) list of active namespaces. And if tolgee is running, loads required data.
     */
    addActiveNs: (ns: import("./types").NsFallback, forget?: boolean | undefined) => Promise<void>;
    /**
     * Remove namespace(s) from active namespaces.
     *
     * Tolgee internally counts how many times was each active namespace added,
     * so this method will remove namespace only if the counter goes down to 0.
     */
    removeActiveNs: (ns: import("./types").NsFallback) => void;
    /**
     * Load records which would be loaded by `run` function
     *
     * You can provide language if not previously set on tolgee instance
     */
    loadRequired: (options?: import("./types").LoadRequiredOptions | undefined) => Promise<import("./types").CacheInternalRecord[]>;
    /**
     * Load records in matrix (languages x namespaces)
     */
    loadMatrix: (options: import("./types").LoadMatrixOptions) => Promise<import("./types").CacheInternalRecord[]>;
    /**
     * Manually load multiple records from `Backend` (or `DevBackend` when in dev mode)
     *
     * It loads data together and adds them to cache in one operation, to prevent partly loaded state.
     */
    loadRecords: (descriptors: import("./types").CacheDescriptor[], options?: import("./types").LoadOptions | undefined) => Promise<import("./types").CacheInternalRecord[]>;
    /**
     * Manually load record from `Backend` (or `DevBackend` when in dev mode)
     */
    loadRecord: (descriptor: import("./types").CacheDescriptor, options?: import("./types").LoadOptions | undefined) => Promise<import("./types").TranslationsFlat>;
    /**
     * Prefill static data
     */
    addStaticData: (data: import("./types").TolgeeStaticDataProp | undefined) => void;
    /**
     * Get record from cache.
     */
    getRecord: (descriptor: import("./types").CacheDescriptor) => import("./types").CacheInternalRecord | undefined;
    /**
     * Get all records from cache.
     */
    getAllRecords: () => (import("./types").CacheInternalRecord | undefined)[];
    /**
     * @param ns optional list of namespaces that you are interested in
     * @return `true` if there are data that need to be fetched.
     */
    isLoaded: (ns?: import("./types").NsFallback) => boolean;
    /**
     * Returns descriptors of records needed for instance to be `loaded`
     */
    getRequiredDescriptors: (lang?: string | undefined, ns?: import("./types").NsFallback) => import("./types").CacheDescriptorInternal[];
    /**
     * @return `true` if tolgee is loading initial data (triggered by `run`).
     */
    isInitialLoading: () => boolean;
    /**
     * @param ns optional list of namespaces that you are interested in
     * @return `true` if tolgee is loading some translations for the first time.
     */
    isLoading: (ns?: import("./types").NsFallback) => boolean;
    /**
     * @param ns optional list of namespaces that you are interested in
     * @return `true` if tolgee is fetching some translations.
     */
    isFetching: (ns?: import("./types").NsFallback) => boolean;
    /**
     * @return `true` if tolgee is running.
     */
    isRunning: () => boolean;
    /**
     * Changes internal state to running: true and loads initial files.
     * Runs runnable plugins mainly Observer if present.
     */
    run: () => Promise<any>;
    /**
     * Changes internal state to running: false and stops runnable plugins.
     */
    stop: () => void;
    /**
     * Returns translated and formatted key.
     * If Observer is present and tolgee is running, wraps result to be identifiable in the DOM.
     */
    t: TFnType<DefaultParamType, string, TranslationKey>;
    /**
     * Highlight keys that match selection.
     */
    highlight: import("./types").HighlightInterface;
    /**
     * Find positions of keys in the DOM.
     */
    findPositions: import("./types").FindPositionsInterface;
    /**
     * @return current Tolgee options.
     */
    getInitialOptions: () => {
        apiUrl?: string | undefined;
        apiKey?: string | undefined;
        projectId?: string | number | undefined;
        language?: string | undefined;
        defaultLanguage?: string | undefined;
        availableLanguages?: string[] | undefined;
        fallbackLanguage?: import("./types").FallbackLanguageOption;
        ns?: string[] | undefined;
        fallbackNs?: import("./types").FallbackGeneral;
        defaultNs?: string | undefined;
        availableNs?: string[] | undefined;
        staticData?: import("./types").TolgeeStaticDataProp | undefined;
        observerType: "invisible" | "text";
        observerOptions: import("./types").ObserverOptionsInternal;
        onFormatError: import("./types").OnFormatError;
        onTranslationMissing: import("./types").MissingTranslationHandler;
        fetch: import("./types").FetchFn;
        tagNewKeys?: string[] | undefined;
        filterTag?: string[] | undefined;
        autoLoadRequiredData: boolean;
    };
    /**
     * Tolgee is in dev mode if `DevTools` plugin is used and `apiKey` + `apiUrl` are specified.
     * @return `true` if tolgee is in dev mode.
     */
    isDev: () => boolean;
    /**
     * Wraps translation if there is `Observer` plugin
     */
    wrap: (params: import("./types").WrapperWrapProps) => string | undefined;
    /**
     * Unwrap translation
     */
    unwrap: (text: string) => import("./types").Unwrapped;
    /**
     * Override creadentials passed on initialization.
     *
     * When called in running state, tolgee stops and runs again.
     */
    overrideCredentials(credentials: DevCredentials): void;
    /**
     * Add tolgee plugin after initialization.
     *
     * When called in running state, tolgee stops and runs again.
     */
    addPlugin(plugin: TolgeePlugin | undefined): void;
    /**
     * Updates options after instance creation. Extends existing options,
     * so it only changes the fields, that are listed.
     *
     * When called in running state, tolgee stops and runs again.
     */
    updateOptions(options?: TolgeeOptions): void;
}>;
export type TolgeeInstance = Omit<ReturnType<typeof createTolgee>, 't'> & {
    t: TFnType<DefaultParamType, string, TranslationKey>;
};
export type TolgeeChainer = {
    /**
     * Add plugin, plugins are applied when `init` method is called.
     */
    use: (plugin: TolgeePlugin | undefined) => TolgeeChainer;
    /**
     * Update default options before tolgee is initialized.
     */
    updateDefaults: (options: TolgeeOptions) => TolgeeChainer;
    /**
     * Initialize tolgee options and apply plugins
     * @returns tolgee instance
     */
    init(options?: TolgeeOptions): TolgeeInstance;
};
/**
 * Tolgee chainable constructor.
 *
 * Usage:
 * ```
 * const tolgee = Tolgee().use(...).init(...)
 * ```
 */
export declare const TolgeeCore: () => TolgeeChainer;
export {};
