function isPromise(value) {
    return Boolean(value && typeof value.then === 'function');
}
function valueOrPromise(value, callback) {
    if (isPromise(value)) {
        return Promise.resolve(value).then(callback);
    }
    else {
        return callback(value);
    }
}
function handleRegularOrAsyncErr(onError, createError, callback) {
    function handle(e) {
        const error = createError(e);
        onError.emit(error);
        // eslint-disable-next-line no-console
        console.error(error);
        throw error;
    }
    try {
        const result = callback();
        if (isPromise(result)) {
            return result.catch(handle);
        }
        return result;
    }
    catch (e) {
        handle(e);
    }
}
function missingOptionError(option) {
    const options = (Array.isArray(option) ? option : [option]).map((val) => `'${val}'`);
    const lastPart = options.slice(-2).join(' or ');
    const firstPart = options.slice(0, -2);
    const stringifiedOptions = [...firstPart, lastPart].join(', ');
    return `Tolgee: You need to specify ${stringifiedOptions} option`;
}
function isObject(item) {
    return typeof item === 'object' && !Array.isArray(item) && item !== null;
}
function getFallback(value) {
    if (typeof value === 'string') {
        return [value];
    }
    if (Array.isArray(value)) {
        return value;
    }
    return undefined;
}
function getFallbackArray(value) {
    return getFallback(value) || [];
}
function getFallbackFromStruct(language, fallbackLanguage) {
    if (isObject(fallbackLanguage)) {
        return getFallbackArray(fallbackLanguage === null || fallbackLanguage === void 0 ? void 0 : fallbackLanguage[language]);
    }
    else {
        return getFallbackArray(fallbackLanguage);
    }
}
function unique(arr) {
    return Array.from(new Set(arr));
}
function sanitizeUrl(url) {
    return url ? url.replace(/\/+$/, '') : url;
}
function getErrorMessage(error) {
    if (typeof error === 'string') {
        return error;
    }
    else if (typeof (error === null || error === void 0 ? void 0 : error.message) === 'string') {
        return error.message;
    }
}
const defaultFetchFunction = (input, options) => fetch(input, options);
function headersInitToRecord(headersInit) {
    return Object.fromEntries(new Headers(headersInit).entries());
}
const createFetchFunction = (fetchFn = defaultFetchFunction) => {
    return (input, init) => {
        let headers = headersInitToRecord(init === null || init === void 0 ? void 0 : init.headers);
        if (headers['x-api-key']) {
            headers = Object.assign({ 'x-tolgee-sdk-type': 'JS', 'x-tolgee-sdk-version': 'prerelease' }, headers);
        }
        return fetchFn(input, Object.assign(Object.assign({}, init), { headers }));
    };
};

const EventEmitter = (type, isActive) => {
    let handlers = [];
    return {
        listen(handler) {
            const handlerWrapper = (e) => {
                handler(e);
            };
            handlers.push(handlerWrapper);
            return {
                unsubscribe() {
                    handlers = handlers.filter((i) => handlerWrapper !== i);
                },
            };
        },
        emit(data) {
            if (isActive()) {
                handlers.forEach((handler) => handler({ type: type, value: data }));
            }
        },
    };
};

function EventEmitterCombined(isActive) {
    let handlers = [];
    let queue = [];
    // merge events in queue into one event
    function solveQueue() {
        if (queue.length === 0) {
            return;
        }
        const queueCopy = queue;
        queue = [];
        handlers.forEach((handler) => {
            handler(queueCopy);
        });
    }
    return Object.freeze({
        listen(handler) {
            const handlerWrapper = (events) => {
                handler(events);
            };
            handlers.push(handlerWrapper);
            return {
                unsubscribe() {
                    handlers = handlers.filter((i) => handlerWrapper !== i);
                },
            };
        },
        emit(e, delayed) {
            if (isActive()) {
                if (isActive()) {
                    queue.push(e);
                    if (!delayed) {
                        solveQueue();
                    }
                    else {
                        setTimeout(solveQueue, 0);
                    }
                }
            }
        },
    });
}

function Events() {
    let emitterActive = true;
    function isActive() {
        return emitterActive;
    }
    const self = Object.freeze({
        onPendingLanguageChange: EventEmitter('pendingLanguage', isActive),
        onLanguageChange: EventEmitter('language', isActive),
        onLoadingChange: EventEmitter('loading', isActive),
        onFetchingChange: EventEmitter('fetching', isActive),
        onInitialLoaded: EventEmitter('initialLoad', isActive),
        onRunningChange: EventEmitter('running', isActive),
        onCacheChange: EventEmitter('cache', isActive),
        onPermanentChange: EventEmitter('permanentChange', isActive),
        onError: EventEmitter('error', isActive),
        onUpdate: EventEmitterCombined(isActive),
        setEmitterActive(active) {
            emitterActive = active;
        },
        on: ((event, handler) => {
            switch (event) {
                case 'pendingLanguage':
                    return self.onPendingLanguageChange.listen(handler);
                case 'language':
                    return self.onLanguageChange.listen(handler);
                case 'loading':
                    return self.onLoadingChange.listen(handler);
                case 'fetching':
                    return self.onFetchingChange.listen(handler);
                case 'initialLoad':
                    return self.onInitialLoaded.listen(handler);
                case 'running':
                    return self.onRunningChange.listen(handler);
                case 'cache':
                    return self.onCacheChange.listen(handler);
                case 'update':
                    return self.onUpdate.listen(handler);
                case 'permanentChange':
                    return self.onPermanentChange.listen(handler);
                case 'error':
                    return self.onError.listen(handler);
            }
        }),
    });
    self.onInitialLoaded.listen((e) => self.onUpdate.emit(e, false));
    self.onLanguageChange.listen((e) => self.onUpdate.emit(e, false));
    self.onCacheChange.listen((e) => self.onUpdate.emit(e, true));
    return self;
}

class RecordFetchError extends Error {
    constructor(descriptor, cause, isDev = false) {
        const { language, namespace } = descriptor;
        super(`Tolgee: Failed to fetch record for "${language}"${namespace && ` and "${namespace}"`}`);
        this.cause = cause;
        this.isDev = isDev;
        this.name = 'RecordFetchError';
        this.language = language;
        this.namespace = namespace;
    }
}
class LanguageDetectorError extends Error {
    constructor(message, cause) {
        super(message);
        this.cause = cause;
        this.name = 'LanguageDetectorError';
    }
}
class LanguageStorageError extends Error {
    constructor(message, cause) {
        super(message);
        this.cause = cause;
        this.name = 'LanguageStorageError';
    }
}

const flattenTranslationsToMap = (data) => {
    const result = new Map();
    Object.entries(data).forEach(([key, value]) => {
        // ignore empty values
        if (value === undefined || value === null) {
            return;
        }
        if (typeof value === 'object') {
            flattenTranslationsToMap(value).forEach((flatValue, flatKey) => {
                result.set(key + '.' + flatKey, flatValue);
            });
            return;
        }
        result.set(key, value);
    });
    return result;
};
const flattenTranslations = (data) => {
    return Object.fromEntries(flattenTranslationsToMap(data).entries());
};
const decodeCacheKey = (key) => {
    const [firstPart, ...rest] = key.split(':');
    // if namespaces contains ":" it won't get lost
    const secondPart = rest.join(':');
    return { language: firstPart, namespace: secondPart || '' };
};
const encodeCacheKey = ({ language, namespace, }) => {
    if (namespace) {
        return `${language}:${namespace}`;
    }
    else {
        return language;
    }
};

function Cache(events, backendGetRecord, backendGetDevRecord, withDefaultNs, isInitialLoading, fetchingObserver, loadingObserver) {
    const asyncRequests = new Map();
    const cache = new Map();
    let staticData = {};
    let version = 0;
    function addRecordInternal(descriptor, data, recordVersion) {
        const cacheKey = encodeCacheKey(descriptor);
        cache.set(cacheKey, {
            data: flattenTranslations(data),
            version: recordVersion,
        });
        events.onCacheChange.emit(decodeCacheKey(cacheKey));
    }
    /**
     * Fetches production data
     */
    async function fetchProd(keyObject) {
        function handleError(e) {
            const error = new RecordFetchError(keyObject, e);
            events.onError.emit(error);
            // eslint-disable-next-line no-console
            console.error(error);
            throw error;
        }
        const dataFromBackend = backendGetRecord(keyObject);
        if (isPromise(dataFromBackend)) {
            const result = await dataFromBackend.catch(handleError);
            if (result !== undefined) {
                return result;
            }
        }
        const staticDataValue = staticData[encodeCacheKey(keyObject)];
        if (typeof staticDataValue === 'function') {
            try {
                return await staticDataValue();
            }
            catch (e) {
                handleError(e);
            }
        }
        else {
            return staticDataValue;
        }
    }
    async function fetchData(keyObject, isDev) {
        let result = undefined;
        if (isDev) {
            try {
                result = await backendGetDevRecord(keyObject);
            }
            catch (e) {
                const error = new RecordFetchError(keyObject, e, true);
                events.onError.emit(error);
                // eslint-disable-next-line no-console
                console.warn(error);
            }
        }
        if (!result) {
            result = await fetchProd(keyObject);
        }
        return result;
    }
    const self = Object.freeze({
        addStaticData(data) {
            if (Array.isArray(data)) {
                for (const record of data) {
                    const key = encodeCacheKey(record);
                    const existing = cache.get(key);
                    if (!existing || existing.version === 0) {
                        addRecordInternal(record, flattenTranslations(record.data), 0);
                    }
                }
            }
            else if (data) {
                staticData = Object.assign(Object.assign({}, staticData), data);
                Object.entries(data).forEach(([key, value]) => {
                    if (typeof value !== 'function') {
                        const descriptor = decodeCacheKey(key);
                        const existing = cache.get(key);
                        if (!existing || existing.version === 0) {
                            addRecordInternal(descriptor, flattenTranslations(value), 0);
                        }
                    }
                });
            }
        },
        invalidate() {
            asyncRequests.clear();
            version += 1;
        },
        addRecord(descriptor, data) {
            addRecordInternal(descriptor, flattenTranslations(data), version);
        },
        exists(descriptor, strict = false) {
            const record = cache.get(encodeCacheKey(descriptor));
            if (record && strict) {
                return record.version === version;
            }
            return Boolean(record);
        },
        getRecord(descriptor) {
            const descriptorWithNs = withDefaultNs(descriptor);
            const cacheKey = encodeCacheKey(descriptorWithNs);
            const cacheRecord = cache.get(cacheKey);
            if (!cacheRecord) {
                return undefined;
            }
            return Object.assign(Object.assign({}, descriptorWithNs), { cacheKey, data: cacheRecord.data });
        },
        getAllRecords() {
            const entries = Array.from(cache.entries());
            return entries.map(([key]) => self.getRecord(decodeCacheKey(key)));
        },
        getTranslation(descriptor, key) {
            var _a;
            return (_a = cache.get(encodeCacheKey(descriptor))) === null || _a === void 0 ? void 0 : _a.data[key];
        },
        getTranslationNs(namespaces, languages, key) {
            var _a;
            for (const namespace of namespaces) {
                for (const language of languages) {
                    const value = (_a = cache.get(encodeCacheKey({ language, namespace }))) === null || _a === void 0 ? void 0 : _a.data[key];
                    if (value !== undefined && value !== null) {
                        return [namespace];
                    }
                }
            }
            return unique(namespaces);
        },
        getTranslationFallback(namespaces, languages, key) {
            var _a;
            for (const namespace of namespaces) {
                for (const language of languages) {
                    const value = (_a = cache.get(encodeCacheKey({ language, namespace }))) === null || _a === void 0 ? void 0 : _a.data[key];
                    if (value !== undefined && value !== null) {
                        return value;
                    }
                }
            }
            return undefined;
        },
        changeTranslation(descriptor, key, value) {
            var _a;
            const record = (_a = cache.get(encodeCacheKey(descriptor))) === null || _a === void 0 ? void 0 : _a.data;
            if (record) {
                record[key] = value;
                events.onCacheChange.emit(Object.assign(Object.assign({}, descriptor), { key }));
            }
        },
        isFetching(ns) {
            if (isInitialLoading()) {
                return true;
            }
            if (ns === undefined) {
                return asyncRequests.size > 0;
            }
            const namespaces = getFallbackArray(ns);
            return Boolean(Array.from(asyncRequests.keys()).find((key) => namespaces.includes(decodeCacheKey(key).namespace)));
        },
        isLoading(language, ns) {
            const namespaces = getFallbackArray(ns);
            if (isInitialLoading()) {
                return true;
            }
            const pendingCacheKeys = Array.from(asyncRequests.keys());
            return Boolean(pendingCacheKeys.find((key) => {
                const descriptor = decodeCacheKey(key);
                return ((!namespaces.length || namespaces.includes(descriptor.namespace)) &&
                    !self.exists({
                        namespace: descriptor.namespace,
                        language: language,
                    }));
            }));
        },
        async loadRecords(descriptors, options) {
            const withPromises = descriptors.map((descriptor) => {
                const keyObject = withDefaultNs(descriptor);
                const cacheKey = encodeCacheKey(keyObject);
                if (options === null || options === void 0 ? void 0 : options.useCache) {
                    const exists = self.exists(keyObject, true);
                    if (exists) {
                        return Object.assign(Object.assign({}, keyObject), { new: false, cacheKey, data: self.getRecord(keyObject).data });
                    }
                }
                const existingPromise = asyncRequests.get(cacheKey);
                if (existingPromise) {
                    return Object.assign(Object.assign({}, keyObject), { new: false, promise: existingPromise, cacheKey });
                }
                const dataPromise = fetchData(keyObject, !(options === null || options === void 0 ? void 0 : options.noDev)) || Promise.resolve(undefined);
                asyncRequests.set(cacheKey, dataPromise);
                return Object.assign(Object.assign({}, keyObject), { new: true, promise: dataPromise, cacheKey });
            });
            fetchingObserver.notify();
            loadingObserver.notify();
            const promisesToWait = withPromises
                .map((val) => val.promise)
                .filter(Boolean);
            const fetchedData = await Promise.all(promisesToWait);
            withPromises.forEach((value) => {
                var _a;
                if (value.promise) {
                    value.data = flattenTranslations((_a = fetchedData[0]) !== null && _a !== void 0 ? _a : {});
                    fetchedData.shift();
                }
                // if promise has changed in between, it means cache been invalidated or
                // new data are being fetched
                const promiseChanged = asyncRequests.get(value.cacheKey) !== value.promise;
                if (value.new && !promiseChanged) {
                    asyncRequests.delete(value.cacheKey);
                    if (value.data) {
                        self.addRecord(value, value.data);
                    }
                    else if (!self.getRecord(value)) {
                        // if no data exist, put empty object
                        // so we know we don't have to fetch again
                        self.addRecord(value, {});
                    }
                }
            });
            fetchingObserver.notify();
            loadingObserver.notify();
            return withPromises.map((val) => {
                var _a;
                return ({
                    language: val.language,
                    namespace: val.namespace,
                    data: (_a = val.data) !== null && _a !== void 0 ? _a : {},
                    cacheKey: val.cacheKey,
                });
            });
        },
    });
    return self;
}

/******************************************************************************
Copyright (c) Microsoft Corporation.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
***************************************************************************** */

function __rest(s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
}

const defaultObserverOptions = {
    tagAttributes: {
        textarea: ['placeholder'],
        input: ['value', 'placeholder'],
        img: ['alt'],
        '*': ['aria-label', 'title'],
    },
    restrictedElements: ['script', 'style'],
    highlightKeys: ['Alt'],
    highlightColor: 'rgb(255, 0, 0)',
    highlightWidth: 5,
    inputPrefix: '%-%tolgee:',
    inputSuffix: '%-%',
    passToParent: ['option', 'optgroup'],
    fullKeyEncode: false,
};

const DEFAULT_FORMAT_ERROR = 'invalid';
const DEFAULT_API_URL = 'https://app.tolgee.io';
const DEFAULT_MISSING_TRANSLATION = ({ key, }) => key;
const defaultValues = {
    observerOptions: defaultObserverOptions,
    observerType: 'invisible',
    onFormatError: DEFAULT_FORMAT_ERROR,
    apiUrl: DEFAULT_API_URL,
    autoLoadRequiredData: true,
    fetch: createFetchFunction(),
    onTranslationMissing: DEFAULT_MISSING_TRANSLATION,
};
const combineOptions = (...states) => {
    let result = {};
    states.forEach((state) => {
        result = Object.assign(Object.assign(Object.assign({}, result), state), { observerOptions: Object.assign(Object.assign({}, result.observerOptions), state === null || state === void 0 ? void 0 : state.observerOptions) });
    });
    return result;
};
function initState(options, previousState) {
    const initialOptions = combineOptions(defaultValues, previousState === null || previousState === void 0 ? void 0 : previousState.initialOptions, options);
    // remove extra '/' from url end
    initialOptions.apiUrl = sanitizeUrl(initialOptions.apiUrl);
    if (options === null || options === void 0 ? void 0 : options.fetch) {
        initialOptions.fetch = createFetchFunction(options.fetch);
    }
    return {
        initialOptions,
        activeNamespaces: (previousState === null || previousState === void 0 ? void 0 : previousState.activeNamespaces) || new Map(),
        language: previousState === null || previousState === void 0 ? void 0 : previousState.language,
        pendingLanguage: previousState === null || previousState === void 0 ? void 0 : previousState.language,
        isInitialLoading: false,
        isRunning: false,
    };
}

function Plugins(getLanguage, getInitialOptions, getAvailableLanguages, getFallbackNamespaces, getTranslationNs, getTranslation, changeTranslation, events) {
    const plugins = {
        ui: undefined,
    };
    const instances = {
        formatters: [],
        finalFormatter: undefined,
        observer: undefined,
        devBackend: undefined,
        backends: [],
        ui: undefined,
        languageDetector: undefined,
        languageStorage: undefined,
    };
    const onClick = async ({ keysAndDefaults, target }) => {
        var _a;
        const withNs = keysAndDefaults.map(({ key, ns, defaultValue }) => {
            return {
                key,
                defaultValue,
                fallbackNamespaces: getFallbackNamespaces(ns),
                namespace: getTranslationNs({ key, ns })[0],
                translation: getTranslation({
                    key,
                    ns,
                }),
            };
        });
        (_a = instances.ui) === null || _a === void 0 ? void 0 : _a.handleElementClick(withNs, target);
    };
    const findPositions = (key, ns) => {
        var _a;
        return ((_a = instances.observer) === null || _a === void 0 ? void 0 : _a.findPositions(key, ns)) || [];
    };
    function translate(props) {
        const translation = getTranslation({
            key: props.key,
            ns: props.ns,
        });
        return self.formatTranslation(Object.assign(Object.assign({}, props), { translation, formatEnabled: true }));
    }
    function getCommonProps() {
        return { fetch: getInitialOptions().fetch };
    }
    function setObserver(observer) {
        instances.observer = observer === null || observer === void 0 ? void 0 : observer();
    }
    function hasObserver() {
        return Boolean(instances.observer);
    }
    function addFormatter(formatter) {
        if (formatter) {
            instances.formatters.push(formatter);
        }
    }
    function setFinalFormatter(formatter) {
        instances.finalFormatter = formatter;
    }
    function setUi(ui) {
        plugins.ui = ui;
    }
    function hasUi() {
        return Boolean(plugins.ui);
    }
    function setLanguageStorage(storage) {
        instances.languageStorage = storage;
    }
    function setLanguageDetector(detector) {
        instances.languageDetector = detector;
    }
    function storageLoadLanguage() {
        return handleRegularOrAsyncErr(events.onError, (e) => new LanguageStorageError('Tolgee: Failed to load language', e), () => { var _a; return (_a = instances.languageStorage) === null || _a === void 0 ? void 0 : _a.getLanguage(getCommonProps()); });
    }
    function detectLanguage() {
        if (!instances.languageDetector) {
            return undefined;
        }
        const availableLanguages = getAvailableLanguages();
        return handleRegularOrAsyncErr(events.onError, (e) => new LanguageDetectorError('Tolgee: Failed to detect language', e), () => {
            var _a;
            return (_a = instances.languageDetector) === null || _a === void 0 ? void 0 : _a.getLanguage(Object.assign({ availableLanguages }, getCommonProps()));
        });
    }
    function addBackend(backend) {
        if (backend) {
            instances.backends.push(backend);
        }
    }
    function setDevBackend(backend) {
        instances.devBackend = backend;
    }
    function addPlugin(tolgeeInstance, plugin) {
        const pluginTools = Object.freeze({
            setFinalFormatter,
            addFormatter,
            setObserver,
            hasObserver,
            setUi,
            hasUi,
            setDevBackend,
            addBackend,
            setLanguageDetector,
            setLanguageStorage,
        });
        plugin(tolgeeInstance, pluginTools);
    }
    const self = Object.freeze({
        addPlugin,
        findPositions: findPositions,
        run() {
            var _a, _b;
            const { apiKey, apiUrl, projectId, observerOptions, tagNewKeys, filterTag, } = getInitialOptions();
            instances.ui = (_a = plugins.ui) === null || _a === void 0 ? void 0 : _a.call(plugins, {
                apiKey: apiKey,
                apiUrl: apiUrl,
                projectId,
                highlight: self.highlight,
                changeTranslation,
                findPositions,
                onPermanentChange: (data) => events.onPermanentChange.emit(data),
                tagNewKeys,
                filterTag,
            });
            (_b = instances.observer) === null || _b === void 0 ? void 0 : _b.run({
                mouseHighlight: Boolean(instances.ui),
                options: observerOptions,
                translate,
                onClick,
            });
        },
        stop() {
            var _a;
            instances.ui = undefined;
            (_a = instances.observer) === null || _a === void 0 ? void 0 : _a.stop();
        },
        getLanguageStorage() {
            return instances.languageStorage;
        },
        getInitialLanguage() {
            const availableLanguages = getAvailableLanguages();
            const languageOrPromise = storageLoadLanguage();
            return valueOrPromise(languageOrPromise, (language) => {
                if ((!availableLanguages || availableLanguages.includes(language)) &&
                    language) {
                    return language;
                }
                return detectLanguage();
            });
        },
        setStoredLanguage(language) {
            return handleRegularOrAsyncErr(events.onError, (e) => new LanguageStorageError('Tolgee: Failed to store language', e), () => { var _a; return (_a = instances.languageStorage) === null || _a === void 0 ? void 0 : _a.setLanguage(language, getCommonProps()); });
        },
        getDevBackend() {
            return instances.devBackend;
        },
        getBackendRecord: (async ({ language, namespace }) => {
            for (const backend of instances.backends) {
                const data = await backend.getRecord(Object.assign({ language,
                    namespace }, getCommonProps()));
                if (data !== undefined) {
                    return data;
                }
            }
            return undefined;
        }),
        getBackendDevRecord: (async ({ language, namespace }) => {
            var _a;
            const { apiKey, apiUrl, projectId, filterTag } = getInitialOptions();
            if (!apiKey || !apiUrl || !self.hasDevBackend()) {
                return undefined;
            }
            return (_a = instances.devBackend) === null || _a === void 0 ? void 0 : _a.getRecord(Object.assign({ apiKey,
                apiUrl,
                projectId,
                language,
                namespace,
                filterTag }, getCommonProps()));
        }),
        getLanguageDetector() {
            return instances.languageDetector;
        },
        retranslate() {
            var _a;
            (_a = instances.observer) === null || _a === void 0 ? void 0 : _a.retranslate();
        },
        highlight: ((key, ns) => {
            var _a, _b;
            return ((_b = (_a = instances.observer) === null || _a === void 0 ? void 0 : _a.highlight) === null || _b === void 0 ? void 0 : _b.call(_a, key, ns)) || { unhighlight() { } };
        }),
        unwrap(text) {
            var _a;
            if (instances.observer) {
                return (_a = instances.observer) === null || _a === void 0 ? void 0 : _a.unwrap(text);
            }
            return { text, keys: [] };
        },
        wrap(params) {
            var _a;
            if (instances.observer) {
                return (_a = instances.observer) === null || _a === void 0 ? void 0 : _a.wrap(params);
            }
            return params.translation;
        },
        hasDevBackend() {
            return Boolean(self.getDevBackend());
        },
        formatTranslation(_a) {
            var _b;
            var { formatEnabled } = _a, props = __rest(_a, ["formatEnabled"]);
            const { key, translation, defaultValue, noWrap, params, ns, orEmpty } = props;
            const formattableTranslation = translation !== null && translation !== void 0 ? translation : defaultValue;
            let translationMissingResult = '';
            if (translation === undefined || translation === null) {
                // called when translation is missing
                // return value is used when 'defaultValue' and 'orEmpty' are not defined
                translationMissingResult =
                    getInitialOptions().onTranslationMissing(props);
            }
            let result = formattableTranslation !== null && formattableTranslation !== void 0 ? formattableTranslation : (orEmpty ? '' : translationMissingResult);
            const language = getLanguage();
            const isFormatEnabled = formatEnabled || !((_b = instances.observer) === null || _b === void 0 ? void 0 : _b.outputNotFormattable);
            const wrap = (result) => {
                if (instances.observer && !noWrap) {
                    return instances.observer.wrap({
                        key,
                        translation: result,
                        defaultValue,
                        params,
                        ns,
                    });
                }
                return result;
            };
            result = wrap(result);
            try {
                if (formattableTranslation && language && isFormatEnabled) {
                    for (const formatter of instances.formatters) {
                        result = formatter.format({
                            translation: result,
                            language,
                            params,
                        });
                    }
                }
                if (instances.finalFormatter &&
                    formattableTranslation &&
                    language &&
                    isFormatEnabled) {
                    result = instances.finalFormatter.format({
                        translation: result,
                        language,
                        params,
                    });
                }
            }
            catch (e) {
                // eslint-disable-next-line no-console
                console.error(e);
                const errorMessage = getErrorMessage(e) || DEFAULT_FORMAT_ERROR;
                const onFormatError = getInitialOptions().onFormatError;
                const formatErrorType = typeof onFormatError;
                if (formatErrorType === 'string') {
                    result = onFormatError;
                }
                else if (formatErrorType === 'function') {
                    result = onFormatError(errorMessage, props);
                }
                else {
                    result = DEFAULT_FORMAT_ERROR;
                }
                // wrap error message, so it's detectable
                result = wrap(result);
            }
            return result;
        },
    });
    return self;
}

const ValueObserver = (initialValue, valueGetter, handler) => {
    let previousValue = initialValue;
    return Object.freeze({
        init(value) {
            previousValue = value;
        },
        notify() {
            const value = valueGetter();
            if (previousValue !== value) {
                handler(value);
            }
            previousValue = value;
        },
    });
};

function State(onLanguageChange, onPendingLanguageChange, onRunningChange) {
    let state = initState();
    let devCredentials = undefined;
    const self = Object.freeze({
        init(options) {
            state = initState(options, state);
        },
        isRunning() {
            return state.isRunning;
        },
        setRunning(value) {
            if (state.isRunning !== value) {
                state.isRunning = value;
                onRunningChange.emit(value);
            }
        },
        isInitialLoading() {
            return state.isInitialLoading;
        },
        setInitialLoading(value) {
            state.isInitialLoading = value;
        },
        getLanguage() {
            return state.language || state.initialOptions.language;
        },
        setLanguage(language) {
            if (state.language !== language) {
                state.language = language;
                onLanguageChange.emit(language);
            }
        },
        getPendingLanguage() {
            return state.pendingLanguage || self.getLanguage();
        },
        setPendingLanguage(language) {
            if (state.pendingLanguage !== language) {
                state.pendingLanguage = language;
                onPendingLanguageChange.emit(language);
            }
        },
        getInitialOptions() {
            return Object.assign(Object.assign({}, state.initialOptions), devCredentials);
        },
        addActiveNs(ns) {
            const namespaces = getFallbackArray(ns);
            namespaces.forEach((namespace) => {
                const value = state.activeNamespaces.get(namespace);
                if (value !== undefined) {
                    state.activeNamespaces.set(namespace, value + 1);
                }
                else {
                    state.activeNamespaces.set(namespace, 1);
                }
            });
        },
        removeActiveNs(ns) {
            const namespaces = getFallbackArray(ns);
            namespaces.forEach((namespace) => {
                const value = state.activeNamespaces.get(namespace);
                if (value !== undefined && value > 1) {
                    state.activeNamespaces.set(namespace, value - 1);
                }
                else {
                    state.activeNamespaces.delete(namespace);
                }
            });
        },
        getRequiredNamespaces() {
            return unique([
                self.getDefaultNs(),
                ...(state.initialOptions.ns || []),
                ...getFallbackArray(state.initialOptions.fallbackNs),
                ...state.activeNamespaces.keys(),
            ]);
        },
        getFallbackLangs(lang) {
            const language = lang || self.getLanguage();
            if (!language) {
                return [];
            }
            return unique([
                language,
                ...getFallbackFromStruct(language, state.initialOptions.fallbackLanguage),
            ]);
        },
        getFallbackNs() {
            return getFallbackArray(state.initialOptions.fallbackNs);
        },
        getNs() {
            var _a, _b;
            return ((_a = state.initialOptions.ns) === null || _a === void 0 ? void 0 : _a.length)
                ? state.initialOptions.ns
                : [(_b = state.initialOptions.defaultNs) !== null && _b !== void 0 ? _b : ''];
        },
        getDefaultNs(ns) {
            var _a, _b, _c;
            return ns === undefined
                ? (_c = (_a = state.initialOptions.defaultNs) !== null && _a !== void 0 ? _a : (_b = state.initialOptions.ns) === null || _b === void 0 ? void 0 : _b[0]) !== null && _c !== void 0 ? _c : ''
                : ns;
        },
        getAvailableLanguages() {
            if (state.initialOptions.availableLanguages) {
                return state.initialOptions.availableLanguages;
            }
            else if (state.initialOptions.staticData) {
                const languagesFromStaticData = Object.keys(state.initialOptions.staticData).map((key) => decodeCacheKey(key).language);
                return Array.from(new Set(languagesFromStaticData));
            }
        },
        getAvailableNs() {
            return state.initialOptions.availableNs;
        },
        withDefaultNs(descriptor) {
            return {
                namespace: descriptor.namespace === undefined
                    ? self.getDefaultNs()
                    : descriptor.namespace,
                language: descriptor.language,
            };
        },
        overrideCredentials(credentials) {
            if (credentials) {
                devCredentials = Object.assign(Object.assign({}, credentials), { apiUrl: sanitizeUrl(credentials.apiUrl) });
            }
            else {
                devCredentials = undefined;
            }
        },
    });
    return self;
}

function parseCombinedOptions(_a) {
    var { ns, noWrap, orEmpty, params, language } = _a, rest = __rest(_a, ["ns", "noWrap", "orEmpty", "params", "language"]);
    const options = {
        ns: ns,
        noWrap: noWrap,
        orEmpty: orEmpty,
        language: language,
    };
    return Object.assign(Object.assign({}, options), { params: Object.assign({}, rest) });
}
const getTranslateProps = (keyOrProps, ...params) => {
    let result = {};
    let options;
    if (typeof keyOrProps === 'object') {
        result = keyOrProps;
    }
    else {
        result.key = keyOrProps;
        if (typeof params[0] === 'string') {
            result.defaultValue = params[0];
            options = params[1];
        }
        else if (typeof params[0] === 'object') {
            options = params[0];
        }
    }
    if (options) {
        result = Object.assign(Object.assign({}, parseCombinedOptions(options)), result);
    }
    return result;
};

function Controller({ options }) {
    const events = Events();
    const fetchingObserver = ValueObserver(false, () => cache.isFetching(), events.onFetchingChange.emit);
    const loadingObserver = ValueObserver(false, () => self.isLoading(), events.onLoadingChange.emit);
    const state = State(events.onLanguageChange, events.onPendingLanguageChange, events.onRunningChange);
    const pluginService = Plugins(state.getLanguage, state.getInitialOptions, state.getAvailableLanguages, getDefaultAndFallbackNs, getTranslationNs, getTranslation, changeTranslation, events);
    const cache = Cache(events, pluginService.getBackendRecord, pluginService.getBackendDevRecord, state.withDefaultNs, state.isInitialLoading, fetchingObserver, loadingObserver);
    if (options) {
        init(options);
    }
    let runPromise;
    events.onUpdate.listen(() => {
        if (state.isRunning()) {
            pluginService.retranslate();
        }
    });
    function getFallbackNs() {
        return state.getFallbackNs();
    }
    function getDefaultNs(ns) {
        return state.getDefaultNs(ns);
    }
    // gets all namespaces where translation could be located
    // takes (ns|default, fallback ns)
    function getDefaultAndFallbackNs(ns) {
        return unique([...getFallbackArray(getDefaultNs(ns)), ...getFallbackNs()]);
    }
    // gets all namespaces which need to be loaded
    // takes (ns|default, initial ns, fallback ns, active ns)
    function getRequiredNamespaces(ns) {
        return unique([
            ...getFallbackArray(ns !== null && ns !== void 0 ? ns : getDefaultNs()),
            ...state.getRequiredNamespaces(),
        ]);
    }
    function changeTranslation(descriptor, key, value) {
        const keyObject = state.withDefaultNs(descriptor);
        const previousValue = cache.getTranslation(keyObject, key);
        cache.changeTranslation(keyObject, key, value);
        return {
            revert() {
                cache.changeTranslation(keyObject, key, previousValue);
            },
        };
    }
    function init(options) {
        state.init(options);
        cache.addStaticData(state.getInitialOptions().staticData);
    }
    function getRequiredDescriptors(lang, ns) {
        const languages = state.getFallbackLangs(lang);
        const namespaces = getRequiredNamespaces(ns);
        const result = [];
        languages.forEach((language) => {
            namespaces.forEach((namespace) => {
                result.push({ language, namespace });
            });
        });
        return result;
    }
    function getMissingDescriptors(lang, ns) {
        return getRequiredDescriptors(lang, ns).filter((descriptor) => !cache.exists(descriptor, true));
    }
    function getMatrixRecords(options) {
        let languages = [];
        let namespaces = [];
        if (Array.isArray(options.languages)) {
            languages = options.languages;
        }
        else if (options.languages === 'all') {
            const availableLanguages = self.getAvailableLanguages();
            if (!availableLanguages) {
                throw new Error(missingOptionError('availableLanguages'));
            }
            languages = availableLanguages;
        }
        if (Array.isArray(options.namespaces)) {
            namespaces = options.namespaces;
        }
        else if (options.namespaces === 'all') {
            const availableNs = self.getAvailableNs();
            if (!availableNs) {
                throw new Error(missingOptionError('availableNs'));
            }
            namespaces = availableNs;
        }
        const records = [];
        languages.forEach((language) => {
            namespaces.forEach((namespace) => {
                records.push({ language, namespace });
            });
        });
        return records;
    }
    function getTranslationNs({ key, ns }) {
        const languages = state.getFallbackLangs();
        const namespaces = getDefaultAndFallbackNs(ns !== null && ns !== void 0 ? ns : undefined);
        return cache.getTranslationNs(namespaces, languages, key);
    }
    function getTranslation({ key, ns, language }) {
        const namespaces = getDefaultAndFallbackNs(ns !== null && ns !== void 0 ? ns : undefined);
        const languages = state.getFallbackLangs(language);
        return cache.getTranslationFallback(namespaces, languages, key);
    }
    function loadInitial() {
        const data = valueOrPromise(initializeLanguage(), () => {
            const missingDescriptors = getMissingDescriptors();
            if (missingDescriptors.length &&
                state.getInitialOptions().autoLoadRequiredData) {
                return cache.loadRecords(missingDescriptors, { useCache: true });
            }
        });
        if (isPromise(data)) {
            state.setInitialLoading(true);
            fetchingObserver.notify();
            loadingObserver.notify();
            return Promise.resolve(data).then(() => {
                state.setInitialLoading(false);
                fetchingObserver.notify();
                loadingObserver.notify();
                events.onInitialLoaded.emit();
            });
        }
        else {
            events.onInitialLoaded.emit();
        }
    }
    function initializeLanguage() {
        const existingLanguage = state.getLanguage();
        if (existingLanguage) {
            return;
        }
        const languageOrPromise = pluginService.getInitialLanguage();
        return valueOrPromise(languageOrPromise, (lang) => {
            const language = lang ||
                state.getInitialOptions().defaultLanguage;
            language && state.setLanguage(language);
        });
    }
    function checkCorrectConfiguration() {
        const languageComputable = pluginService.getLanguageDetector() || pluginService.getLanguageStorage();
        if (languageComputable) {
            const availableLanguages = state.getAvailableLanguages();
            if (!availableLanguages) {
                throw new Error(missingOptionError('availableLanguages'));
            }
        }
        if (!state.getLanguage() && !state.getInitialOptions().defaultLanguage) {
            throw new Error(missingOptionError(['defaultLanguage', 'language']));
        }
    }
    const self = Object.freeze(Object.assign(Object.assign(Object.assign(Object.assign(Object.assign({}, events), state), pluginService), cache), { init: init, getTranslation: getTranslation, changeTranslation: changeTranslation, getTranslationNs: getTranslationNs, getDefaultAndFallbackNs: getDefaultAndFallbackNs, findPositions: pluginService.findPositions, getRequiredDescriptors: getRequiredDescriptors, async changeLanguage(language) {
            if (state.getPendingLanguage() === language &&
                state.getLanguage() === language) {
                return;
            }
            state.setPendingLanguage(language);
            if (state.isRunning() && state.getInitialOptions().autoLoadRequiredData) {
                await cache.loadRecords(getRequiredDescriptors(language), {
                    useCache: true,
                });
            }
            if (language === state.getPendingLanguage()) {
                // there might be parallel language change
                // we only want to apply latest
                state.setLanguage(language);
                await pluginService.setStoredLanguage(language);
            }
        },
        async addActiveNs(ns, forget) {
            if (!forget) {
                state.addActiveNs(ns);
            }
            if (state.isRunning()) {
                await cache.loadRecords(getRequiredDescriptors(undefined, ns), {
                    useCache: true,
                });
            }
        },
        async loadRecord(descriptor, options) {
            var _a;
            return (_a = (await self.loadRecords([descriptor], options))[0]) === null || _a === void 0 ? void 0 : _a.data;
        },
        isLoading(ns) {
            return cache.isLoading(state.getLanguage(), ns);
        },
        isLoaded(ns) {
            const language = state.getLanguage();
            if (!language) {
                return false;
            }
            const languages = state.getFallbackLangs(language);
            const namespaces = getRequiredNamespaces(ns);
            const result = [];
            languages.forEach((language) => {
                namespaces.forEach((namespace) => {
                    if (!cache.exists({ language, namespace })) {
                        result.push({ language, namespace });
                    }
                });
            });
            return result.length === 0;
        }, t: ((...args) => {
            // @ts-ignore
            const params = getTranslateProps(...args);
            const translation = getTranslation(params);
            return pluginService.formatTranslation(Object.assign(Object.assign({}, params), { translation }));
        }), isDev() {
            return Boolean(state.getInitialOptions().apiKey && state.getInitialOptions().apiUrl);
        },
        async loadRequired(options) {
            if (!(options === null || options === void 0 ? void 0 : options.language)) {
                await initializeLanguage();
            }
            const requiredRecords = getRequiredDescriptors(options === null || options === void 0 ? void 0 : options.language);
            return self.loadRecords(requiredRecords, options);
        },
        async loadMatrix(options) {
            const records = getMatrixRecords(options);
            return self.loadRecords(records, options);
        },
        run() {
            checkCorrectConfiguration();
            if (!state.isRunning()) {
                state.setRunning(true);
                pluginService.run();
                runPromise = loadInitial();
            }
            return Promise.resolve(runPromise);
        },
        stop() {
            if (state.isRunning()) {
                pluginService.stop();
                state.setRunning(false);
            }
        } }));
    return self;
}

function createTolgee(options) {
    const controller = Controller({
        options,
    });
    if (controller.isDev()) {
        // override existing data in DevMode
        controller.invalidate();
    }
    // restarts tolgee while applying callback
    function withRestart(callback) {
        const wasRunning = controller.isRunning();
        wasRunning && controller.stop();
        callback();
        // invalidate cache when tolgee configuration is updated/plugin added in DevMode
        controller.isDev() && controller.invalidate();
        wasRunning && controller.run();
    }
    const self = Object.freeze({
        /**
         * Listen to tolgee events.
         */
        on: controller.on,
        /**
         * Turn off/on events emitting. Is on by default.
         */
        setEmitterActive: controller.setEmitterActive,
        /**
         * @return current language if set.
         */
        getLanguage: controller.getLanguage,
        /**
         * `pendingLanguage` represents language which is currently being loaded.
         * @return current `pendingLanguage` if set.
         */
        getPendingLanguage: controller.getPendingLanguage,
        /**
         * Change current language.
         * - if not running sets `pendingLanguage`, `language` to the new value
         * - if running sets `pendingLanguage` to the value, fetches necessary data and then changes `language`
         *
         * @return Promise which is resolved when `language` is changed.
         */
        changeLanguage: controller.changeLanguage,
        /**
         * Temporarily change translation in cache.
         * @return object with revert method.
         */
        changeTranslation: controller.changeTranslation,
        /**
         * Adds namespace(s) list of active namespaces. And if tolgee is running, loads required data.
         */
        addActiveNs: controller.addActiveNs,
        /**
         * Remove namespace(s) from active namespaces.
         *
         * Tolgee internally counts how many times was each active namespace added,
         * so this method will remove namespace only if the counter goes down to 0.
         */
        removeActiveNs: controller.removeActiveNs,
        /**
         * Load records which would be loaded by `run` function
         *
         * You can provide language if not previously set on tolgee instance
         */
        loadRequired: controller.loadRequired,
        /**
         * Load records in matrix (languages x namespaces)
         */
        loadMatrix: controller.loadMatrix,
        /**
         * Manually load multiple records from `Backend` (or `DevBackend` when in dev mode)
         *
         * It loads data together and adds them to cache in one operation, to prevent partly loaded state.
         */
        loadRecords: controller.loadRecords,
        /**
         * Manually load record from `Backend` (or `DevBackend` when in dev mode)
         */
        loadRecord: controller.loadRecord,
        /**
         * Prefill static data
         */
        addStaticData: controller.addStaticData,
        /**
         * Get record from cache.
         */
        getRecord: controller.getRecord,
        /**
         * Get all records from cache.
         */
        getAllRecords: controller.getAllRecords,
        /**
         * @param ns optional list of namespaces that you are interested in
         * @return `true` if there are data that need to be fetched.
         */
        isLoaded: controller.isLoaded,
        /**
         * Returns descriptors of records needed for instance to be `loaded`
         */
        getRequiredDescriptors: controller.getRequiredDescriptors,
        /**
         * @return `true` if tolgee is loading initial data (triggered by `run`).
         */
        isInitialLoading: controller.isInitialLoading,
        /**
         * @param ns optional list of namespaces that you are interested in
         * @return `true` if tolgee is loading some translations for the first time.
         */
        isLoading: controller.isLoading,
        /**
         * @param ns optional list of namespaces that you are interested in
         * @return `true` if tolgee is fetching some translations.
         */
        isFetching: controller.isFetching,
        /**
         * @return `true` if tolgee is running.
         */
        isRunning: controller.isRunning,
        /**
         * Changes internal state to running: true and loads initial files.
         * Runs runnable plugins mainly Observer if present.
         */
        run: controller.run,
        /**
         * Changes internal state to running: false and stops runnable plugins.
         */
        stop: controller.stop,
        /**
         * Returns translated and formatted key.
         * If Observer is present and tolgee is running, wraps result to be identifiable in the DOM.
         */
        t: controller.t,
        /**
         * Highlight keys that match selection.
         */
        highlight: controller.highlight,
        /**
         * Find positions of keys in the DOM.
         */
        findPositions: controller.findPositions,
        /**
         * @return current Tolgee options.
         */
        getInitialOptions: controller.getInitialOptions,
        /**
         * Tolgee is in dev mode if `DevTools` plugin is used and `apiKey` + `apiUrl` are specified.
         * @return `true` if tolgee is in dev mode.
         */
        isDev: controller.isDev,
        /**
         * Wraps translation if there is `Observer` plugin
         */
        wrap: controller.wrap,
        /**
         * Unwrap translation
         */
        unwrap: controller.unwrap,
        /**
         * Override creadentials passed on initialization.
         *
         * When called in running state, tolgee stops and runs again.
         */
        overrideCredentials(credentials) {
            withRestart(() => controller.overrideCredentials(credentials));
        },
        /**
         * Add tolgee plugin after initialization.
         *
         * When called in running state, tolgee stops and runs again.
         */
        addPlugin(plugin) {
            if (plugin) {
                withRestart(() => controller.addPlugin(self, plugin));
            }
        },
        /**
         * Updates options after instance creation. Extends existing options,
         * so it only changes the fields, that are listed.
         *
         * When called in running state, tolgee stops and runs again.
         */
        updateOptions(options) {
            if (options) {
                withRestart(() => controller.init(options));
            }
        },
    });
    return self;
}
/**
 * Tolgee chainable constructor.
 *
 * Usage:
 * ```
 * const tolgee = Tolgee().use(...).init(...)
 * ```
 */
const TolgeeCore = () => {
    const state = {
        plugins: [],
        options: {},
    };
    const tolgeeChain = Object.freeze({
        use(plugin) {
            state.plugins.push(plugin);
            return tolgeeChain;
        },
        updateDefaults(options) {
            state.options = combineOptions(state.options, options);
            return tolgeeChain;
        },
        init(options) {
            const tolgee = createTolgee(combineOptions(state.options, options));
            state.plugins.forEach(tolgee.addPlugin);
            return tolgee;
        },
    });
    return tolgeeChain;
};

const ERROR_PARAM_EMPTY = 0, ERROR_UNEXPECTED_CHAR = 1, ERROR_UNEXPECTED_END = 2;
class FormatError extends Error {
    constructor(code, index, text) {
        let error;
        let hint = '';
        if (code === ERROR_PARAM_EMPTY) {
            error = 'Empty parameter';
        }
        else if (code === ERROR_UNEXPECTED_CHAR) {
            error = 'Unexpected character';
            hint = 'Did you forget to use FormatIcu to render ICU message syntax?';
        }
        else {
            error = 'Unexpected end';
        }
        super(`Tolgee parser: ${error} at ${index} in "${text}"` +
            (hint ? '\n' + hint : ''));
        this.code = code;
        this.index = index;
    }
}

function isWhitespace(ch) {
    return /\s/.test(ch);
}
const STATE_TEXT = 0, STATE_ESCAPE_MAYBE = 1, STATE_ESCAPE = 2, STATE_PARAM = 3, STATE_PARAM_AFTER = 4;
const END_STATES = new Set([
    STATE_ESCAPE,
    STATE_ESCAPE_MAYBE,
    STATE_TEXT,
]);
const CHAR_ESCAPE = "'";
const ESCAPABLE = new Set(['{', '}', CHAR_ESCAPE]);
const isAllowedInParam = (char) => {
    return /[0-9a-zA-Z_]/.test(char);
};
function formatParser(translation) {
    let state = STATE_TEXT;
    let text = '';
    let param = '';
    let ch = '';
    const texts = [];
    const params = [];
    let i = 0;
    function parsingError(code) {
        throw new FormatError(code, i, translation);
    }
    const addText = () => {
        texts.push(text);
        text = '';
    };
    const addParamChar = () => {
        if (!isAllowedInParam(ch)) {
            parsingError(ERROR_UNEXPECTED_CHAR);
        }
        param += ch;
    };
    const addParam = () => {
        if (param === '') {
            parsingError(ERROR_PARAM_EMPTY);
        }
        params.push(param);
        param = '';
    };
    for (i = 0; i < translation.length; i++) {
        ch = translation[i];
        switch (state) {
            case STATE_TEXT:
                if (ch === CHAR_ESCAPE) {
                    text += ch;
                    state = STATE_ESCAPE_MAYBE;
                }
                else if (ch === '{') {
                    addText();
                    state = STATE_PARAM;
                }
                else {
                    text += ch;
                    state = STATE_TEXT;
                }
                break;
            case STATE_ESCAPE_MAYBE:
                if (ESCAPABLE.has(ch)) {
                    text = text.slice(0, -1) + ch;
                    state = STATE_ESCAPE;
                }
                else {
                    text += ch;
                    state = STATE_TEXT;
                }
                break;
            case STATE_ESCAPE:
                if (ch === CHAR_ESCAPE) {
                    state = STATE_TEXT;
                }
                else {
                    text += ch;
                    state = STATE_ESCAPE;
                }
                break;
            case STATE_PARAM:
                if (ch === '}') {
                    addParam();
                    state = STATE_TEXT;
                }
                else if (!isWhitespace(ch)) {
                    addParamChar();
                    state = STATE_PARAM;
                }
                else if (param !== '') {
                    addParam();
                    state = STATE_PARAM_AFTER;
                }
                break;
            case STATE_PARAM_AFTER:
                if (ch == '}') {
                    state = STATE_TEXT;
                }
                else if (isWhitespace(ch)) {
                    state = STATE_PARAM_AFTER;
                }
                else {
                    parsingError(ERROR_UNEXPECTED_CHAR);
                }
        }
    }
    if (!END_STATES.has(state)) {
        parsingError(ERROR_UNEXPECTED_END);
    }
    addText();
    return [texts, params];
}

function formatter(translation, params) {
    const [texts, pars] = formatParser(translation);
    const result = [texts[0]];
    for (let i = 1; i < texts.length; i++) {
        const parameter = params === null || params === void 0 ? void 0 : params[pars[i - 1]];
        if (parameter === undefined) {
            throw new Error(`Missing parameter "${pars[i - 1]}" in "${translation}"`);
        }
        result.push(String(parameter));
        result.push(texts[i]);
    }
    return result.join('');
}

function createFormatSimple() {
    return {
        format: ({ translation, params }) => formatter(translation, params),
    };
}
const FormatSimple = () => (tolgee, tools) => {
    tools.setFinalFormatter(createFormatSimple());
    return tolgee;
};

export { FormatSimple, LanguageDetectorError, LanguageStorageError, RecordFetchError, TolgeeCore, createFetchFunction, getFallback, getFallbackArray, getTranslateProps };
//# sourceMappingURL=tolgee.esm.js.map
