var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { extname } from 'path';
import { readFile } from 'fs/promises';
import { createRequire } from 'module';
import TextMate from 'vscode-textmate';
import Oniguruma from 'vscode-oniguruma';
const GRAMMAR_PATH = new URL('../../textmate/', import.meta.url);
const GrammarFiles = {
    ["source.ts" /* Grammar.TYPESCRIPT */]: new URL('TypeScript.tmLanguage', GRAMMAR_PATH),
    ["source.tsx" /* Grammar.TYPESCRIPT_TSX */]: new URL('TypeScriptReact.tmLanguage', GRAMMAR_PATH),
    ["source.svelte" /* Grammar.SVELTE */]: new URL('Svelte.tmLanguage', GRAMMAR_PATH),
    ["source.vue" /* Grammar.VUE */]: new URL('Vue.tmLanguage', GRAMMAR_PATH),
    ["html-template.ng" /* Grammar.ANGULAR_HTML */]: new URL('AngularHtml.tmLanguage', GRAMMAR_PATH),
    ["text.html.basic" /* Grammar.HTML */]: new URL('HTML.tmLanguage', GRAMMAR_PATH),
    ["text.html.derivative" /* Grammar.HTML_D */]: new URL('HTML.tmLanguage', GRAMMAR_PATH),
};
let oniguruma;
let registry;
function initializeOniguruma() {
    return __awaiter(this, void 0, void 0, function* () {
        const require = createRequire(import.meta.url);
        const wasmBlobPath = require
            .resolve('vscode-oniguruma')
            .replace('main.js', 'onig.wasm');
        const wasmBlob = yield readFile(wasmBlobPath);
        yield Oniguruma.loadWASM(wasmBlob);
        return {
            createOnigScanner: (patterns) => new Oniguruma.OnigScanner(patterns),
            createOnigString: (s) => new Oniguruma.OnigString(s),
        };
    });
}
function loadGrammar(scope) {
    return __awaiter(this, void 0, void 0, function* () {
        const file = GrammarFiles[scope];
        if (!file)
            return null;
        const grammar = yield readFile(file, 'utf8');
        return JSON.parse(grammar);
    });
}
function fileNameToGrammar(fileName) {
    const ext = extname(fileName);
    switch (ext) {
        case '.js':
        case '.mjs':
        case '.cjs':
        case '.ts':
        case '.mts':
        case '.cts':
            return "source.ts" /* Grammar.TYPESCRIPT */;
        case '.jsx':
        case '.tsx':
            return "source.tsx" /* Grammar.TYPESCRIPT_TSX */;
        case '.vue':
            return "source.vue" /* Grammar.VUE */;
        case '.svelte':
            return "source.svelte" /* Grammar.SVELTE */;
        case '.html':
            if (/.*\.component\.html/.test(fileName)) {
                return "html-template.ng" /* Grammar.ANGULAR_HTML */;
            }
            else {
                return "text.html.basic" /* Grammar.HTML */;
            }
    }
}
function tokenize(code, grammar) {
    let stack = TextMate.INITIAL;
    let linePtr = 0;
    const lines = code.split('\n');
    const tokens = [];
    for (let i = 0; i < lines.length; i++) {
        const line = lines[i];
        const res = grammar.tokenizeLine(line, stack);
        for (const token of res.tokens) {
            const codeToken = code.slice(linePtr + token.startIndex, linePtr + token.endIndex);
            // Opinionated take: if a token is scope-less and void, chances are we don't care about it.
            // Ditching it allows us to reduce complexity from the state machine's POV.
            if (token.scopes.length !== 1 || codeToken.trim()) {
                tokens.push({
                    type: token.scopes[token.scopes.length - 1],
                    token: codeToken,
                    line: i + 1,
                    startIndex: linePtr + token.startIndex,
                    endIndex: linePtr + token.endIndex,
                    scopes: token.scopes,
                });
            }
        }
        tokens.push({
            type: 'newline',
            token: '\n',
            line: i + 1,
            startIndex: linePtr + line.length,
            endIndex: linePtr + line.length + 1,
            scopes: [],
        });
        linePtr += line.length + 1;
        stack = res.ruleStack;
    }
    return tokens;
}
export default function (code, fileName) {
    return __awaiter(this, void 0, void 0, function* () {
        if (!oniguruma) {
            // Lazily initialize the WebAssembly runtime
            oniguruma = initializeOniguruma();
            registry = new TextMate.Registry({
                onigLib: oniguruma,
                loadGrammar: loadGrammar,
            });
        }
        const grammarName = fileNameToGrammar(fileName);
        if (!grammarName) {
            throw new Error(`Cannot find grammar for file type ${fileName}`);
        }
        const grammar = yield registry.loadGrammar(grammarName);
        if (!grammar) {
            throw new Error(`Cannot load grammar ${grammarName}`);
        }
        return tokenize(code, grammar);
    });
}
