var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { extname, join } from 'path';
import { readdir, readFile, stat } from 'fs/promises';
import { Command, Option } from 'commander';
import { glob } from 'tinyglobby';
import { exit } from 'process';
import { loading, success, error, warn, exitWithError, } from '../utils/logger.js';
import { askString } from '../utils/ask.js';
import { mapImportFormat } from '../utils/mapImportFormat.js';
import { handleLoadableError } from '../client/TolgeeClient.js';
import { findFilesByTemplate } from '../utils/filesTemplate.js';
import { valueToArray } from '../utils/valueToArray.js';
import { printUnresolvedConflicts } from '../utils/printFailedKeys.js';
function allInPattern(pattern) {
    return __awaiter(this, void 0, void 0, function* () {
        const files = [];
        const items = yield glob(pattern);
        for (const item of items) {
            if ((yield stat(item)).isDirectory()) {
                files.push(...(yield readDirectory(item)));
            }
            else {
                const blob = yield readFile(item);
                files.push({ name: item, data: blob });
            }
        }
        return files;
    });
}
function readDirectory(directory_1) {
    return __awaiter(this, arguments, void 0, function* (directory, base = '') {
        const files = [];
        const dir = yield readdir(directory);
        for (const file of dir) {
            const filePath = join(directory, file);
            const fileStat = yield stat(filePath);
            if (fileStat.isDirectory()) {
                const dirFiles = yield readDirectory(filePath, `${file}/`);
                files.push(...dirFiles);
            }
            else {
                const blob = yield readFile(filePath);
                files.push({ name: base + file, data: blob });
            }
        }
        return files;
    });
}
function promptConflicts(opts) {
    return __awaiter(this, void 0, void 0, function* () {
        if (opts.forceMode === 'NO_FORCE') {
            exitWithError(`There are conflicts in the import and the force mode is set to "NO_FORCE". Set it to "KEEP" or "OVERRIDE" to continue.`);
        }
        if (opts.forceMode) {
            return opts.forceMode;
        }
        if (!process.stdout.isTTY) {
            exitWithError(`There are conflicts in the import. Please specify a --force-mode.`);
        }
        warn('There are conflicts in the import. What do you want to do?');
        const resp = yield askString('Type "KEEP" to preserve the version on the server, "OVERRIDE" to use the version from the import, and nothing to abort: ');
        if (resp !== 'KEEP' && resp !== 'OVERRIDE') {
            exitWithError(`Aborting.`);
        }
        return resp;
    });
}
function importData(client, data) {
    return __awaiter(this, void 0, void 0, function* () {
        return loading('Uploading files...', client.import.import(data));
    });
}
function readRecords(matchers) {
    return __awaiter(this, void 0, void 0, function* () {
        const result = [];
        for (const matcher of matchers) {
            const files = yield allInPattern(matcher.path);
            files.forEach((file) => {
                result.push(Object.assign(Object.assign({}, matcher), { data: file.data, name: file.name }));
            });
        }
        return result;
    });
}
function handleMappingError(fileMappings) {
    error('Not able to map files to existing languages in the platform');
    console.log(`Pushed files:`);
    fileMappings.forEach(({ fileName, languageTag }) => {
        console.log(`"${fileName}"${languageTag ? ` => "${languageTag}"` : ''}`);
    });
    console.log('\nYou can use `push.files[*].language` property in `tolgeerc` file to map language correctly');
    exit(1);
}
const pushHandler = (config) => function () {
    return __awaiter(this, void 0, void 0, function* () {
        var _a, _b, _c, _d, _e, _f, _g;
        const opts = this.optsWithGlobals();
        let allMatchers = [];
        const filesTemplate = opts.filesTemplate;
        if (!filesTemplate && !((_b = (_a = config.push) === null || _a === void 0 ? void 0 : _a.files) === null || _b === void 0 ? void 0 : _b.length)) {
            exitWithError('Missing option `push.filesTemplate` or `push.files`.');
        }
        if (filesTemplate) {
            for (const template of filesTemplate) {
                allMatchers = allMatchers.concat(...(yield findFilesByTemplate(template)));
            }
        }
        allMatchers = allMatchers.concat(...(((_c = config.push) === null || _c === void 0 ? void 0 : _c.files) || []));
        const filteredMatchers = allMatchers.filter((r) => {
            var _a;
            if (r.language &&
                opts.languages &&
                !opts.languages.includes(r.language)) {
                return false;
            }
            if (opts.namespaces && !opts.namespaces.includes((_a = r.namespace) !== null && _a !== void 0 ? _a : '')) {
                return false;
            }
            return true;
        });
        const files = yield loading('Reading files...', readRecords(filteredMatchers));
        if (files.length === 0) {
            error('Nothing to import.');
            return;
        }
        let errorOnUnresolvedConflict;
        switch (opts.errorOnUnresolvedConflict) {
            case 'auto':
                errorOnUnresolvedConflict = undefined;
                break;
            case 'yes':
                errorOnUnresolvedConflict = true;
                break;
            case 'no':
                errorOnUnresolvedConflict = false;
                break;
        }
        const params = {
            createNewKeys: true,
            forceMode: opts.forceMode,
            overrideKeyDescriptions: opts.overrideKeyDescriptions,
            convertPlaceholdersToIcu: opts.convertPlaceholdersToIcu,
            tagNewKeys: (_d = opts.tagNewKeys) !== null && _d !== void 0 ? _d : [],
            overrideMode: (_e = opts.overrideMode) !== null && _e !== void 0 ? _e : 'RECOMMENDED',
            fileMappings: files.map((f) => {
                var _a;
                const format = mapImportFormat(opts.format, extname(f.name));
                return {
                    fileName: f.name,
                    format: format,
                    languageTag: f.language,
                    namespace: (_a = f.namespace) !== null && _a !== void 0 ? _a : '',
                    // there can be multiple languages inside the file
                    // that is detected on the backend
                    // and we only say which we want to import by `languageTagsToImport`
                    // however we don't want to use this property,
                    // when it's explicitly defined what language is in the file
                    languageTagsToImport: !f.language ? opts.languages : undefined,
                };
            }),
            removeOtherKeys: opts.removeOtherKeys,
            errorOnUnresolvedConflict: errorOnUnresolvedConflict,
        };
        let attempt = yield loading('Importing...', importData(opts.client, {
            files,
            params,
        }));
        if (attempt.error) {
            if (attempt.error.code === 'existing_language_not_selected') {
                handleMappingError(params.fileMappings);
            }
            if (attempt.error.code !== 'conflict_is_not_resolved') {
                handleLoadableError(attempt);
            }
            const forceMode = yield promptConflicts(opts);
            attempt = yield loading('Overriding...', importData(opts.client, {
                files,
                params: Object.assign(Object.assign({}, params), { forceMode }),
            }));
            handleLoadableError(attempt);
        }
        if ((_g = (_f = attempt.data) === null || _f === void 0 ? void 0 : _f.unresolvedConflicts) === null || _g === void 0 ? void 0 : _g.length) {
            printUnresolvedConflicts(attempt.data.unresolvedConflicts, false);
        }
        success('Done!');
    });
};
export default (config) => {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
    return new Command()
        .name('push')
        .description('Pushes translations to Tolgee')
        .addOption(new Option('-ft, --files-template <templates...>', 'A template that describes the structure of the local files and their location with file structure template format (more at: https://docs.tolgee.io/tolgee-cli/push-pull-strings#file-structure-template-format).\n\nExample: `./public/{namespace}/{languageTag}.json`\n\n').default(valueToArray((_a = config.push) === null || _a === void 0 ? void 0 : _a.filesTemplate)))
        .addOption(new Option('-f, --force-mode <mode>', 'What should we do with possible conflicts? If unspecified, the user will be prompted interactively, or the command will fail when in non-interactive')
        .choices(['OVERRIDE', 'KEEP', 'NO_FORCE'])
        .argParser((v) => v.toUpperCase())
        .default((_b = config.push) === null || _b === void 0 ? void 0 : _b.forceMode))
        .addOption(new Option('--override-key-descriptions', 'Override existing key descriptions from local files (only relevant for some formats).').default((_d = (_c = config.push) === null || _c === void 0 ? void 0 : _c.overrideKeyDescriptions) !== null && _d !== void 0 ? _d : true))
        .addOption(new Option('--convert-placeholders-to-icu', 'Convert placeholders in local files to ICU format.').default((_f = (_e = config.push) === null || _e === void 0 ? void 0 : _e.convertPlaceholdersToIcu) !== null && _f !== void 0 ? _f : true))
        .addOption(new Option('-l, --languages <languages...>', 'Specifies which languages should be pushed (see push.files in config).').default((_g = config.push) === null || _g === void 0 ? void 0 : _g.languages))
        .addOption(new Option('-n, --namespaces <namespaces...>', 'Specifies which namespaces should be pushed (see push.files in config).').default((_h = config.push) === null || _h === void 0 ? void 0 : _h.namespaces))
        .addOption(new Option('--tag-new-keys <tags...>', 'Specify tags that will be added to newly created keys.').default((_j = config.push) === null || _j === void 0 ? void 0 : _j.tagNewKeys))
        .addOption(new Option('--remove-other-keys', 'Remove keys which are not present in the import (within imported namespaces).').default((_k = config.push) === null || _k === void 0 ? void 0 : _k.removeOtherKeys))
        .addOption(new Option('--override-mode <mode>', 'Specifies what is considered non-overridable translation.')
        .choices(['RECOMMENDED', 'ALL'])
        .default((_l = config.push) === null || _l === void 0 ? void 0 : _l.overrideMode))
        .addOption(new Option('--error-on-unresolved-conflict <choice>', 'Fail the whole import if there are unresolved conflicts.')
        .choices(['yes', 'no', 'auto'])
        .default((_o = (_m = config.push) === null || _m === void 0 ? void 0 : _m.errorOnUnresolvedConflict) !== null && _o !== void 0 ? _o : 'auto'))
        .action(pushHandler(config));
};
