use axum::extract::Path;
use axum::routing::{get, post};
use axum::{Json, Router};

use crate::roll::{Roll, RollType};
use crate::roll_statistics::RollStatistics;
use crate::server::backend::AppState;
use crate::{
    name_generator::{Genders, NameGenerators, get_generator},
    random_features::RandomFeatures,
    server::cache_response::CacheResponse,
};

pub fn get_router() -> Router<AppState> {
    Router::new()
        .route(
            "/name-generator/{race}/genders",
            get(get_name_generator_genders),
        )
        .route(
            "/name-generator/roll/{n_rolls}/{race}/{gender}",
            post(roll_name_generator),
        )
        .route(
            "/random-features/roll/{n}/{positive}",
            post(get_random_features),
        )
        .route(
            "/roll-statistics/{n_roll}/{number_dices}/{roll_type}/{miserable}/{weary}",
            get(roll_statistics),
        )
}

async fn get_name_generator_genders(
    Path(race): Path<NameGenerators>,
) -> CacheResponse<Vec<Genders>> {
    let generator = get_generator(race);
    CacheResponse::new(generator.get_genders())
}

// Use post to prevent any caching
async fn roll_name_generator(
    Path((n_rolls, race, gender)): Path<(u32, NameGenerators, Genders)>,
) -> Json<Vec<String>> {
    let generator = get_generator(race);
    let mut out = vec![];
    for _ in 0..n_rolls {
        out.push(generator.get_name(gender));
    }
    Json(out)
}

// Use post to prevent any caching
async fn get_random_features(Path((n, positive)): Path<(u32, bool)>) -> Json<Vec<String>> {
    let mut out = vec![];
    for _ in 0..n {
        out.push(RandomFeatures::get(positive));
    }
    Json(out)
}

async fn roll_statistics(
    Path((n_roll, number_dices, roll_type, miserable, weary)): Path<(
        u32,
        u32,
        RollType,
        bool,
        bool,
    )>,
) -> CacheResponse<RollStatistics> {
    let mut stats = RollStatistics::new();
    for _ in 0..n_roll {
        stats += Roll::roll_dices(roll_type, number_dices, weary, miserable);
    }

    stats.finalize();
    CacheResponse::new(stats)
}
