use anyhow::anyhow;
use axum::extract::{Path, State};
use axum::routing::{delete, post, put};
use axum::{Json, Router};
use enums::character_type::CharacterType;

use crate::character::Character;
use crate::character::advantage::AdvantageTrait as _;
use crate::character::item::item_trait::ItemTrait as _;
use crate::character::item::{Item, ItemType};
use crate::character::item::{armor::Armor, equipment::Equipment, shield::Shield, weapon::Weapon};
use crate::error::Result;
use crate::server::backend::AppState;
use crate::server::users::User;

pub fn get_router() -> Router<AppState> {
    axum::Router::new()
        .route("/{character_id}/item", put(add_item))
        .route("/{character_id}/{item_type}/{id}", delete(delete_item))
        .route(
            "/{character_id}/{item_type}/wearing/{id}/{value}",
            put(update_wearing),
        )
        .route(
            "/{character_id}/link/{reward_id}/{item_type}/{item_id}",
            post(link_reward).delete(unlink_reward),
        )
}

async fn add_item(
    state: State<AppState>,
    user: User,
    Path(character_id): Path<i32>,
    Json(item): Json<Item>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.items.add_item(&item)?;

    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn delete_item(
    state: State<AppState>,
    user: User,
    Path((character_id, item_type, id)): Path<(i32, ItemType, i32)>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.items.delete_item(item_type, id)?;
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

async fn update_wearing(
    state: State<AppState>,
    user: User,
    Path((character_id, item_type, id, value)): Path<(i32, ItemType, i32, bool)>,
) -> Result<()> {
    let mut character = Character::get(&state.db, &user, character_id).await?;
    character.items.update_wearing(item_type, id, value)?;
    character.save_to_db(&state.db, &user).await?;
    Ok(())
}

fn check_allow_link(
    character: &Character,
    reward_id: i32,
    item_type: ItemType,
    item_id: i32,
) -> Result<()> {
    if character.get_character_type() == CharacterType::External {
        Err(anyhow!("Cannot link a reward to an external character"))?;
    }

    if !character
        .characteristics
        .get_rewards()
        .iter()
        .map(|reward| reward.get_id().expect("Reward should have an id here"))
        .collect::<Vec<_>>()
        .contains(&reward_id)
    {
        Err(anyhow!(
            "You cannot link to a reward that does not belong to you"
        ))?;
    }
    if !character.items.get_items_id(item_type).contains(&item_id) {
        Err(anyhow!(
            "You cannot link to an item that does not belong to you"
        ))?;
    }

    Ok(())
}

async fn link_reward(
    state: State<AppState>,
    user: User,
    Path((character_id, reward_id, item_type, item_id)): Path<(i32, i32, ItemType, i32)>,
) -> Result<()> {
    let character = Character::get(&state.db, &user, character_id).await?;

    check_allow_link(&character, reward_id, item_type, item_id)?;

    match item_type {
        ItemType::Armor => Armor::link_reward(&state.db, item_id, reward_id).await?,
        ItemType::Equipment => Equipment::link_reward(&state.db, item_id, reward_id).await?,
        ItemType::Weapon => Weapon::link_reward(&state.db, item_id, reward_id).await?,
        ItemType::Shield => Shield::link_reward(&state.db, item_id, reward_id).await?,
    }
    Ok(())
}

async fn unlink_reward(
    state: State<AppState>,
    user: User,
    Path((character_id, reward_id, item_type, item_id)): Path<(i32, i32, ItemType, i32)>,
) -> Result<()> {
    let character = Character::get(&state.db, &user, character_id).await?;

    check_allow_link(&character, reward_id, item_type, item_id)?;

    match item_type {
        ItemType::Armor => Armor::unlink_reward(&state.db, reward_id).await?,
        ItemType::Equipment => Equipment::unlink_reward(&state.db, reward_id).await?,
        ItemType::Weapon => Weapon::unlink_reward(&state.db, reward_id).await?,
        ItemType::Shield => Shield::unlink_reward(&state.db, reward_id).await?,
    }
    Ok(())
}
