use serde::{Deserialize, Serialize};
use std::cmp;
use ts_rs::TS;

/// Ensure that derived is set to the right value
#[derive(Default, Serialize, Deserialize, TS, Debug, Clone)]
struct InternalStateCap {
    #[serde(default)]
    current: u32,
    max: u32,
}

#[derive(Default, Serialize, Deserialize, TS, Debug, Clone)]
#[ts(export)]
#[serde(from = "InternalStateCap")]
pub struct StateCap {
    current: u32,
    max: u32,
    computed_max: u32,
}

impl From<InternalStateCap> for StateCap {
    fn from(value: InternalStateCap) -> Self {
        Self {
            current: value.current,
            max: value.max,
            computed_max: value.max,
        }
    }
}

impl StateCap {
    pub fn new(max: u32) -> Self {
        Self {
            current: max,
            max,
            computed_max: max,
        }
    }

    pub fn set_computed_value(&mut self, value: u32) {
        self.computed_max = value;
    }
    pub fn set_current(&mut self, current: u32) {
        self.current = cmp::min(self.computed_max, current);
    }

    pub fn loose(&mut self, value: i32) {
        self.current = match self.current.checked_add_signed(-value) {
            Some(val) => val,
            None => {
                if value > 0 {
                    0
                } else {
                    self.computed_max
                }
            }
        };
        self.set_current(self.current);
    }

    pub fn get_max(&self) -> u32 {
        self.max
    }

    pub fn get_computed_max(&self) -> u32 {
        self.computed_max
    }

    pub fn get_current(&self) -> u32 {
        self.current
    }

    pub fn reset(&mut self) {
        self.current = self.computed_max;
    }
}
