use std::convert::Into;
use std::fmt::Debug;

use anyhow::{Context as _, Result, bail};
use enums::character_data::CharacterData;
use enums::virtue::VirtueEnum;
use sea_orm::ActiveValue::Set;
use sea_orm::QueryFilter as _;
use sea_orm::{
    ActiveModelTrait as _, ColumnTrait as _, ConnectionTrait, EntityTrait, IntoActiveModel,
    ModelTrait as _, PrimaryKeyTrait,
};
use std::collections::HashSet;

use super::Item;
use super::equipment::Equipment;
use super::{armor::Armor, shield::Shield, weapon::Weapon};
use crate::character::advantage::AdvantageTrait as _;
use crate::character::reward::Reward;
use crate::common::computed_value::ComputedValue;

pub trait RewardItem: EntityTrait + Send {
    fn get_reward_column() -> Self::Column;
    fn get_item_column() -> Self::Column;
    fn get_reward_id(model: &Self::Model) -> i32;
    fn get_item_id(model: &Self::Model) -> i32;
}

pub trait ItemEntity: EntityTrait {
    fn set_character_id(value: &mut Self::ActiveModel, character_id: i32);
    fn get_character_column() -> Self::Column;
    fn get_character_id(model: &Self::Model) -> i32;
    fn get_id(model: &Self::Model) -> i32;
}

#[expect(async_fn_in_trait)]
pub trait ItemTrait:
    Sized + Debug + Clone + From<<Self::Entity as sea_orm::EntityTrait>::Model>
{
    type Entity: ItemEntity<
            Model: IntoActiveModel<<Self::Entity as EntityTrait>::ActiveModel>,
            ActiveModel: From<Self> + Send,
            PrimaryKey: PrimaryKeyTrait<ValueType = i32>,
        >;
    type RewardItem: RewardItem<
            Model: IntoActiveModel<<Self::RewardItem as EntityTrait>::ActiveModel>,
            ActiveModel: Send,
        >;

    fn compute_derived_values(&mut self, data: &CharacterData);

    fn get_item(&self) -> Item;

    fn set_rewards(&mut self, rewards: Vec<Reward>);

    fn is_famous(&self) -> bool;

    fn set_id(&mut self, id: Option<i32>);

    fn get_name(&self) -> &str;

    fn get_id(&self) -> Option<i32>;

    fn get_load(&self, virtues: &[VirtueEnum]) -> ComputedValue<u32>;

    fn is_wearing(&self) -> bool;

    fn set_wearing(&mut self, wearing: bool);

    fn get_mut_rewards(&mut self) -> &mut [Reward];

    fn get_reward_item_model(
        item_id: i32,
        reward_id: i32,
    ) -> <Self::RewardItem as EntityTrait>::ActiveModel;

    // Do not implement the functions below
    async fn rewards_from_db<C: ConnectionTrait>(
        &self,
        db: &C,
        character_id: i32,
    ) -> Result<Vec<Reward>> {
        let ids: HashSet<i32> = <Self::RewardItem as EntityTrait>::find()
            .filter(
                Self::RewardItem::get_item_column().eq(self
                    .get_id()
                    .context("Cannot fetch rewards for an item without an id")?),
            )
            .all(db)
            .await?
            .into_iter()
            .map(|model| Self::RewardItem::get_reward_id(&model))
            .collect();

        Ok(Reward::from_db_model(db, character_id)
            .await?
            .into_iter()
            .filter(|reward| {
                ids.contains(&reward.get_id().expect("Rewards should have an id here"))
            })
            .collect())
    }

    async fn from_db_reward<C: ConnectionTrait>(db: &C, reward_id: i32) -> Result<Option<Self>> {
        let item = <Self::RewardItem as EntityTrait>::find()
            .filter(Self::RewardItem::get_reward_column().eq(reward_id))
            .one(db)
            .await?;

        let out = if let Some(item) = item {
            let id = Self::RewardItem::get_item_id(&item);
            Self::Entity::find_by_id(id).one(db).await?.map(Into::into)
        } else {
            None
        };

        Ok(out)
    }

    async fn link_already_exist_all<C: ConnectionTrait>(db: &C, reward_id: i32) -> Result<bool> {
        if Armor::link_already_exist_single(db, reward_id).await? {
            return Ok(true);
        }
        if Equipment::link_already_exist_single(db, reward_id).await? {
            return Ok(true);
        }
        if Weapon::link_already_exist_single(db, reward_id).await? {
            return Ok(true);
        }
        Shield::link_already_exist_single(db, reward_id).await
    }

    // WARNING: Does not check ownership
    async fn unlink_reward<C: ConnectionTrait>(db: &C, reward_id: i32) -> Result<()> {
        <Self::RewardItem as EntityTrait>::delete_many()
            .filter(Self::RewardItem::get_reward_column().eq(reward_id))
            .exec(db)
            .await?;
        Ok(())
    }

    // WARNING: Does not check ownership
    async fn link_reward<C: ConnectionTrait>(db: &C, item_id: i32, reward_id: i32) -> Result<()> {
        if Self::link_already_exist_all(db, reward_id).await? {
            bail!("Cannot link a reward multiple times")
        }
        Self::get_reward_item_model(item_id, reward_id)
            .insert(db)
            .await?;
        Ok(())
    }

    async fn link_already_exist_single<C: ConnectionTrait>(db: &C, reward_id: i32) -> Result<bool> {
        let result = <Self::RewardItem as EntityTrait>::find()
            .filter(Self::RewardItem::get_reward_column().eq(reward_id))
            .one(db)
            .await?;
        Ok(result.is_some())
    }

    async fn save<C: ConnectionTrait>(&self, db: &C, character_id: i32) -> Result<()> {
        let mut model: <Self::Entity as EntityTrait>::ActiveModel = (*self).clone().into();
        Self::Entity::set_character_id(&mut model, character_id);
        model.save(db).await?;
        Ok(())
    }

    async fn from_db_model<C: ConnectionTrait>(db: &C, character_id: i32) -> Result<Vec<Self>> {
        let mut items: Vec<Self> = Self::Entity::find()
            .filter(Self::Entity::get_character_column().eq(character_id))
            .all(db)
            .await?
            .iter()
            .map(|x| (*x).clone().into())
            .collect();

        for item in &mut items {
            item.set_rewards(item.rewards_from_db(db, character_id).await?);
        }

        Ok(items)
    }

    // WARNING does not check ownership
    async fn delete_by_character_id<C: ConnectionTrait>(db: &C, character_id: i32) -> Result<()> {
        let items = Self::from_db_model(db, character_id)
            .await?
            .into_iter()
            .map(|item| item.get_id().expect("Item should have an id here"));
        Self::RewardItem::delete_many()
            .filter(Self::RewardItem::get_item_column().is_in(items))
            .exec(db)
            .await?;
        Self::Entity::delete_many()
            .filter(Self::Entity::get_character_column().eq(character_id))
            .exec(db)
            .await?;
        Ok(())
    }

    async fn delete_by_id<C: ConnectionTrait>(db: &C, character_id: i32, id: i32) -> Result<()> {
        let eq = Self::Entity::find_by_id(id).one(db).await?;

        if let Some(eq) = eq {
            if Self::Entity::get_character_id(&eq) == character_id {
                let item_id = Self::Entity::get_id(&eq);
                let link = Self::RewardItem::find()
                    .filter(Self::RewardItem::get_item_column().eq(item_id))
                    .one(db)
                    .await?;
                if link.is_some() {
                    Self::RewardItem::delete_many()
                        .filter(Self::RewardItem::get_item_column().eq(item_id))
                        .exec(db)
                        .await?;
                }
                eq.delete(db).await?;
                Ok(())
            } else {
                bail!("Cannot delete item from someone else")
            }
        } else {
            bail!("Item not found")
        }
    }

    // Insert all missing links between rewards and items
    async fn update_links<C: ConnectionTrait>(
        db: &C,
        character_id: i32,
        items: &mut [Self],
        rewards: &mut [Reward],
    ) -> Result<()> {
        for item in items.iter_mut() {
            // Find the matches between items and rewards
            let item_id = item.get_id().expect("Item should have an id here");
            let item_object = item.get_item();
            for item_reward in item.get_mut_rewards() {
                for reward in rewards.iter_mut() {
                    if item_reward.get_title() == reward.get_title() {
                        Self::link_reward(
                            db,
                            item_id,
                            reward.get_id().expect("Reward should have an id here"),
                        )
                        .await?;
                        *item_reward = reward.clone();
                        reward.set_linked_item(item_object.clone());
                    }
                }
            }
        }
        // Delete links that do not exist anymore
        for item in items {
            let in_db = item.rewards_from_db(db, character_id).await?;
            for reward in in_db {
                if !item
                    .get_mut_rewards()
                    .iter()
                    .any(|x| x.get_id() == reward.get_id())
                {
                    Self::unlink_reward(
                        db,
                        reward.get_id().expect("Reward should have an id here"),
                    )
                    .await?;
                }
            }
        }
        Ok(())
    }

    async fn update_ids<C: ConnectionTrait>(
        db: &C,
        character_id: i32,
        items: &mut [Self],
    ) -> Result<()> {
        let mut in_db = Self::from_db_model(db, character_id).await?;

        for item in items {
            if item.get_id().is_none() {
                bail!("Id is null. This should not happen.")
            }
            // Id matches nothing to do
            if let Some(position) = in_db.iter().position(|item2| {
                item.get_id() == item2.get_id() && item.get_name() == item2.get_name()
            }) {
                in_db.remove(position);
                continue;
            }

            // Find same name
            if let Some(position) = in_db
                .iter()
                .position(|item2| item.get_name() == item2.get_name())
            {
                let id = in_db
                    .remove(position)
                    .get_id()
                    .context("Id is not set. This should not happen")?;
                item.set_id(Some(id));
                continue;
            }

            // Not found so reset id
            item.set_id(None);
        }

        // Remove all the items that do not exist anymore
        for item in in_db {
            <Self as ItemTrait>::delete_by_id(
                db,
                character_id,
                item.get_id().expect("Item should have an id here"),
            )
            .await?;
        }

        Ok(())
    }
}

pub mod macros {
    use super::{ItemEntity, RewardItem, Set};
    use crate::entity::{
        armor, equipment, reward_armor, reward_equipment, reward_shield, reward_weapon, shield,
        weapon,
    };

    macro_rules! impl_reward_item {
        ($tp: ty, $item_column: ident, $item_value: ident) => {
            impl RewardItem for $tp {
                fn get_reward_column() -> Self::Column {
                    Self::Column::RewardId
                }
                fn get_item_column() -> Self::Column {
                    Self::Column::$item_column
                }
                fn get_reward_id(model: &Self::Model) -> i32 {
                    model.reward_id
                }
                fn get_item_id(model: &Self::Model) -> i32 {
                    model.$item_value
                }
            }
        };
    }

    macro_rules! impl_item_entity {
        ($tp: ty) => {
            impl ItemEntity for $tp {
                fn get_character_id(model: &Self::Model) -> i32 {
                    model.character_id
                }
                fn get_character_column() -> Self::Column {
                    Self::Column::CharacterId
                }
                fn set_character_id(value: &mut Self::ActiveModel, character_id: i32) {
                    value.character_id = Set(character_id);
                }
                fn get_id(model: &Self::Model) -> i32 {
                    model.id
                }
            }
        };
    }

    #[macro_export]
    macro_rules! impl_item_trait {
        ($tp: ident, $reward_item: ty, $entity: ty, $item_value: ident, $active_model: path) => {
            use std::cmp::max;
            use $crate::character::advantage::AdvantageTrait as _;

            impl $crate::character::item::ItemTrait for $tp {
                type RewardItem = $reward_item;
                type Entity = $entity;

                fn get_reward_item_model(
                    item_id: i32,
                    reward_id: i32,
                ) -> <Self::RewardItem as sea_orm::EntityTrait>::ActiveModel {
                    $active_model {
                        reward_id: Set(reward_id),
                        $item_value: Set(item_id),
                    }
                }

                fn compute_derived_values(&mut self, data: &enums::character_data::CharacterData) {
                    let diff = self.rewards.iter().fold(0, |acc, reward| {
                        acc + if let Some(enum_value) = reward.get_enum() {
                            enum_value.decrease_load(data)
                        } else {
                            0
                        }
                    });

                    self.load.set_computed_value(max(0, self.load.value - diff));
                    self.compute_special_derived_values(data);
                }

                fn get_mut_rewards(&mut self) -> &mut [Reward] {
                    &mut self.rewards
                }

                fn get_item(&self) -> $crate::character::item::Item {
                    $crate::character::item::Item::$tp(self.clone())
                }

                fn set_rewards(&mut self, rewards: Vec<$crate::character::reward::Reward>) {
                    self.rewards = rewards;
                }

                fn is_famous(&self) -> bool {
                    self.famous
                }

                fn get_name(&self) -> &str {
                    &self.name
                }

                fn set_id(&mut self, id: Option<i32>) {
                    self.id = id;
                }

                fn get_id(&self) -> Option<i32> {
                    self.id
                }

                fn get_load(&self, _: &[enums::virtue::VirtueEnum]) -> ComputedValue<u32> {
                    self.load
                }

                fn is_wearing(&self) -> bool {
                    self.wearing
                }

                fn set_wearing(&mut self, wearing: bool) {
                    self.wearing = wearing;
                }
            }
        };
    }

    impl_reward_item!(reward_equipment::Entity, EquipmentId, equipment_id);
    impl_reward_item!(reward_weapon::Entity, WeaponId, weapon_id);
    impl_reward_item!(reward_shield::Entity, ShieldId, shield_id);
    impl_reward_item!(reward_armor::Entity, ArmorId, armor_id);

    impl_item_entity!(equipment::Entity);
    impl_item_entity!(weapon::Entity);
    impl_item_entity!(armor::Entity);
    impl_item_entity!(shield::Entity);
}
