use std::vec;

use enums::armor::ArmorEnum;
use enums::character_data::CharacterData;
use sea_orm::ActiveValue::{NotSet, Set};
use serde::{Deserialize, Serialize};
use ts_rs::TS;

use crate::assets::read_enum_from_assets;
use crate::character::reward::Reward;
use crate::common::computed_value::ComputedValue;
use crate::common::enums::find_enum;
use crate::entity::armor as entity;
use crate::entity::reward_armor;
use crate::impl_item_trait;
use crate::utils::convert_option_to_set;
use enums::reward::RewardEnum;

const DIRECTORY: &str = "items/armors";

#[derive(Serialize, Deserialize, Clone, Default, TS, Debug)]
#[ts(export)]
pub struct Armor {
    id: Option<i32>,
    name: String,
    protection: u32,
    enum_value: Option<ArmorEnum>,
    load: ComputedValue<u32>,
    #[serde(default)]
    wearing: bool,
    #[serde(default)]
    notes: String,
    #[serde(default)]
    famous: bool,
    #[serde(default)]
    rewards: Vec<Reward>,
}

impl Armor {
    pub fn new(name: String, protection: u32, load: u32) -> Self {
        Armor {
            id: None,
            name,
            protection,
            load: ComputedValue::new(load),
            wearing: true,
            notes: String::new(),
            famous: false,
            rewards: vec![],
            enum_value: None,
        }
    }

    pub fn get_protection(&self) -> u32 {
        self.protection
    }
    #[deprecated(since = "3.0.0", note = "Please use the sql functions")]
    pub fn reset_id(&mut self) {
        self.id = None;
    }
    fn compute_special_derived_values(&mut self, _data: &CharacterData) {
        let load_diff = if self
            .rewards
            .iter()
            .any(|rew| rew.get_enum() == Some(RewardEnum::MithrilArmour))
            && let Some(armor) = self.enum_value
        {
            match armor {
                ArmorEnum::CoatOfMail => 6,
                ArmorEnum::MailShirt => 3,
                _ => 0,
            }
        } else if self
            .rewards
            .iter()
            .any(|rew| rew.get_enum() == Some(RewardEnum::MithrilHelm))
            && let Some(armor) = self.enum_value
        {
            match armor {
                ArmorEnum::Helm => 4,
                _ => 0,
            }
        } else {
            0
        };
        self.load
            .set_computed_value(max(0, self.load.value - load_diff));
    }
}

impl From<ArmorEnum> for Armor {
    fn from(value: ArmorEnum) -> Self {
        let mut out: Self = read_enum_from_assets(&value, DIRECTORY);
        out.enum_value = Some(value);
        out
    }
}

impl From<&str> for Armor {
    fn from(value: &str) -> Self {
        let value = format!("\"{value}\"");
        let value: ArmorEnum =
            serde_json::from_str(&value).expect("Failed to convert to armor enum");
        Self::from(value)
    }
}

impl_item_trait!(
    Armor,
    reward_armor::Entity,
    entity::Entity,
    armor_id,
    reward_armor::ActiveModel
);

impl From<entity::Model> for Armor {
    fn from(value: entity::Model) -> Self {
        let enum_value = value.enum_value.as_ref().and_then(|name| {
            serde_json::from_str(name).expect("SQL and serde enums are not compatible")
        });
        Self {
            id: Some(value.id),
            enum_value: match enum_value {
                Some(x) => Some(x),
                None => find_enum(&value.name),
            },
            name: value.name,
            protection: value.protection.try_into().expect("Failed to convert"),
            load: ComputedValue::new(value.load.try_into().expect("Failed to convert")),
            wearing: value.wearing,
            notes: value.notes,
            famous: value.famous,
            rewards: vec![],
        }
    }
}

impl From<Armor> for entity::ActiveModel {
    fn from(value: Armor) -> Self {
        let enum_value = value
            .enum_value
            .as_ref()
            .map(|value| serde_json::to_string(value).expect("Failed to convert enum to string"));
        let id = convert_option_to_set(value.id);
        Self {
            id,
            name: Set(value.name),
            protection: Set(value.protection.try_into().expect("Failed to convert")),
            load: Set(value.load.value.try_into().expect("Failed to convert")),
            wearing: Set(value.wearing),
            notes: Set(value.notes),
            character_id: NotSet,
            famous: Set(value.famous),
            enum_value: Set(enum_value),
        }
    }
}

#[cfg(test)]
mod test {
    use strum::{EnumCount as _, IntoEnumIterator as _};

    use crate::assets::get_number_files;

    use super::*;

    #[test]
    fn check_armor_data() {
        let number_files = get_number_files(DIRECTORY);
        assert!(ArmorEnum::COUNT == number_files);

        for armor in ArmorEnum::iter() {
            eprintln!("Testing {armor:?}");
            let _: Armor = Armor::from(armor);
        }
    }
}
