import { defaultResultPage, DriverService, KNOWN_SOFTWARE, PostParser, } from '@dhaaga/bridge';
import { TimelineFetchMode } from '@dhaaga/core';
import { queryOptions } from '@tanstack/react-query';
export function unifiedPostFeedQueryOptions(client, driver, server, acctIdentifier, { type, query, opts, maxId, minId, sessionId, limit = 15, }) {
    const _id = query?.id;
    const _query = {
        ...opts,
        limit: limit,
        sinceId: minId,
        untilId: maxId === null ? undefined : maxId,
        maxId,
        minId, // quirks
        userId: _id,
    };
    function _parseCursorData(data, maxId) {
        if (maxId)
            return maxId;
        if (typeof data === 'object' &&
            !Array.isArray(data) &&
            Object.hasOwn(data, 'maxId') &&
            data.maxId != null) {
            return data.maxId;
        }
        if (Array.isArray(data)) {
            return data.length > 0 ? (data[data.length - 1].id ?? null) : null;
        }
        if (Array.isArray(data.data)) {
            return data.data.length > 0
                ? (data.data[data.data.length - 1].id ?? null)
                : null;
        }
        throw new Error('unknown cursor format for returned feed');
    }
    function _parseFeedData(data) {
        if (data.data) {
            return PostParser.parse(data.data, driver, server);
        }
        else if (Array.isArray(data)) {
            return PostParser.parse(data, driver, server);
        }
        else {
            throw new Error('unknown data format for returned feed');
        }
    }
    function createResultBatch(data, maxId) {
        return {
            data: _parseFeedData(data),
            maxId: _parseCursorData(data, maxId),
            minId: null,
        };
    }
    async function api() {
        switch (type) {
            case TimelineFetchMode.IDLE:
                return defaultResultPage;
            case TimelineFetchMode.HOME: {
                const result = await client.timelines.home(_query);
                return createResultBatch(result.data, result.maxId);
            }
            case TimelineFetchMode.LOCAL: {
                const result = await client.timelines.public({
                    ..._query,
                    local: true, // Pleroma/Akkoma thing
                    withMuted: DriverService.supportsPleromaApi(driver)
                        ? true
                        : undefined,
                    withRenotes: !opts?.excludeReblogs,
                    withReplies: !opts?.excludeReplies,
                });
                return createResultBatch(result.data, result.maxId);
            }
            case TimelineFetchMode.HASHTAG: {
                const data = await client.timelines.hashtag(_id, _query);
                return createResultBatch(data.data, data.maxId);
            }
            case TimelineFetchMode.LIST: {
                const result = await client.timelines.list(_id, _query);
                return createResultBatch(result);
            }
            case TimelineFetchMode.USER: {
                if (!_query || _query.userId === undefined)
                    throw new Error('missing userId');
                const data = await client.users.getPosts(_id, _query);
                return createResultBatch(data.data, data.maxId);
            }
            case TimelineFetchMode.SOCIAL: {
                const result = await client.timelines.public({
                    ..._query,
                    social: true,
                });
                return createResultBatch(result);
            }
            case TimelineFetchMode.BUBBLE: {
                if (DriverService.supportsPleromaApi(driver)) {
                    const result = await client.timelines.bubble(_query);
                    return {
                        data: PostParser.parse(result.data, driver, server),
                        maxId: _parseCursorData(result.data),
                        minId: null,
                    };
                }
                else if (driver === KNOWN_SOFTWARE.SHARKEY) {
                    const result = await client.timelines.bubble(_query);
                    return {
                        data: PostParser.parse(result, driver, server),
                        maxId: _parseCursorData(result),
                        minId: null,
                    };
                }
                else {
                    return defaultResultPage;
                }
            }
            case TimelineFetchMode.FEDERATED: {
                const data = await client.timelines.public(_query);
                return createResultBatch(data.data, data.maxId);
            }
            case TimelineFetchMode.BOOKMARKS: {
                const data = await client.users.bookmarks(_query);
                return createResultBatch(data.data, data?.maxId);
            }
            case TimelineFetchMode.FEED: {
                const data = await client.timelines.getFeed({
                    limit,
                    cursor: maxId === null ? undefined : maxId,
                    feed: query.id,
                });
                return createResultBatch(data.data, data.maxId);
            }
            case TimelineFetchMode.LIKES: {
                if (DriverService.supportsAtProto(driver)) {
                    const data = await client.users.atProtoLikes(acctIdentifier, {
                        limit,
                        cursor: _query.maxId === null ? undefined : _query.maxId,
                    });
                    return createResultBatch(data.data, data.maxId);
                }
                const data = await client.users.likes(_query);
                return createResultBatch(data.data, data.maxId);
            }
            case TimelineFetchMode.TRENDING_POSTS: {
                if (!DriverService.supportsMastoApiV2(driver)) {
                    throw new Error('trending posts only supported on Mastodon v2 API compatible servers');
                }
                const data = await client.trends.posts({
                    limit: 20,
                    offset: 0,
                });
                return createResultBatch(data);
            }
            case TimelineFetchMode.TRENDING_USERS: {
                if (!DriverService.supportsMastoApiV2(driver)) {
                    throw new Error('trending posts only supported on Mastodon v2 API compatible servers');
                }
                const data = await client.trends.posts({
                    limit: 20,
                    offset: 0,
                });
                return createResultBatch(data);
            }
            case TimelineFetchMode.TRENDING_TAGS: {
                if (!DriverService.supportsMastoApiV2(driver)) {
                    throw new Error('trending posts only supported on Mastodon v2 API compatible servers');
                }
                const data = await client.trends.posts({
                    limit: 20,
                    offset: 0,
                });
                return createResultBatch(data);
            }
            default: {
                throw new Error(`unknown timeline type: ${type}`);
            }
        }
    }
    return queryOptions({
        queryKey: [
            'dhaaga/feed/unified/posts',
            type,
            _id,
            _query,
            maxId,
            minId,
            sessionId,
        ],
        queryFn: api,
        enabled: !!client && type !== TimelineFetchMode.IDLE,
        initialData: defaultResultPage,
    });
}
//# sourceMappingURL=post-feed.js.map