import { ActivityPubService, GroupedNotificationParser, ChatParser, DriverService, } from '@dhaaga/bridge';
import { queryOptions } from '@tanstack/react-query';
const NOTIFICATION_PAGE_SIZE = 20;
function getMentionNotificationsQueryOpts(client, maxId) {
    async function api() {
        const results = await client.notifications.getMentions({
            limit: NOTIFICATION_PAGE_SIZE,
            maxId,
        });
        if (ActivityPubService.misskeyLike(client.driver)) {
            return GroupedNotificationParser.parseForMisskey(results.data, client.driver, client.server);
        }
        else if (ActivityPubService.supportsV2(client.driver)) {
            return GroupedNotificationParser.parseForMastodonV2(results, client.driver, client.server);
        }
        else if (ActivityPubService.pleromaLike(client.driver)) {
            return GroupedNotificationParser.parseForMastodonV1(results.data, client.driver, client.server, 'mentions');
        }
        else if (ActivityPubService.blueskyLike(client.driver)) {
            return GroupedNotificationParser.parseForBluesky(results, client, client.driver, client.server);
        }
        else {
            throw new Error('unsupported driver');
        }
    }
    return queryOptions({
        queryKey: ['dhaaga/inbox/mentions', client?.key, maxId],
        queryFn: api,
        enabled: client !== null,
    });
}
function getChatNotificationsQueryOpts(client) {
    async function api() {
        const result = await client.notifications.getChats(client.driver);
        return {
            ...result,
            data: ChatParser.parse(result.data, client),
        };
    }
    return queryOptions({
        queryKey: ['dhaaga/inbox/chat', client?.key],
        queryFn: api,
        enabled: !!client,
    });
}
function getSocialNotificationsQueryOpts(client, maxId) {
    async function api() {
        const result = await client.notifications.getSocialUpdates({
            limit: NOTIFICATION_PAGE_SIZE,
            maxId,
        });
        if (ActivityPubService.misskeyLike(client.driver)) {
            return GroupedNotificationParser.parseForMisskey(result, client.driver, client.server);
        }
        else if (ActivityPubService.supportsV2(client.driver)) {
            return GroupedNotificationParser.parseForMastodonV2(result, client.driver, client.server);
        }
        else if (ActivityPubService.pleromaLike(client.driver)) {
            return GroupedNotificationParser.parseForMastodonV1(result, client.driver, client.server, 'social');
        }
        else if (DriverService.supportsAtProto(client.driver)) {
            return GroupedNotificationParser.parseForBluesky(result, client, client.driver, client.server);
        }
        else {
            throw new Error('unsupported driver');
        }
    }
    return queryOptions({
        queryKey: ['dhaaga/inbox/social', client?.key, maxId],
        queryFn: api,
        enabled: !!client,
    });
}
function getSubscriptionNotificationsQueryOpts(client, maxId) {
    async function api() {
        const result = await client.notifications.getSubscriptions(maxId);
        if (DriverService.supportsMisskeyApi(client.driver)) {
            return GroupedNotificationParser.parseForMisskey(result, client.driver, client.server);
        }
        else if (DriverService.supportsMastoApiV2(client.driver)) {
            return GroupedNotificationParser.parseForMastodonV2(result, client.driver, client.server);
        }
        else if (DriverService.supportsPleromaApi(client.driver)) {
            return GroupedNotificationParser.parseForMastodonV1(result, client.driver, client.server, 'updates');
        }
        else if (DriverService.supportsAtProto(client.driver)) {
            return GroupedNotificationParser.parseForBluesky(result, client, client.driver, client.server);
        }
        else {
            throw new Error(`unsupported driver ${client.driver}`);
        }
    }
    return queryOptions({
        queryKey: ['dhaaga/inbox/updates', client?.key, maxId],
        queryFn: api,
        enabled: !!client,
    });
}
export { getMentionNotificationsQueryOpts, getChatNotificationsQueryOpts, getSocialNotificationsQueryOpts, getSubscriptionNotificationsQueryOpts, };
//# sourceMappingURL=notifications.js.map