import { DbErrorHandler } from './_base.repo.js';
import { DataSource } from '../dataSource.js';
import { Account, Profile } from '../_schema.js';
import { eq, gt } from '@dhaaga/orm';
import { AccountService } from './account.js';
import { ProfilePinnedTimelineService } from './profile-pinned-timeline.js';
import { ProfilePinnedTagService } from './profile-pinned-tag.js';
import { RandomUtil } from '@dhaaga/bridge';

@DbErrorHandler()
class Repo {}

class Service {
	static getShownProfiles(db: DataSource) {
		return db.profile.find({ active: true });
	}

	static renameProfileById(
		db: DataSource,
		id: number | string,
		newName: string,
	) {
		const profile = Service.getById(db, id);
		if (profile) {
			db.profile.updateById(profile.id, {
				name: newName,
			});
		}
	}

	static isDefaultProfile(db: DataSource, profile: Profile) {
		return profile.uuid === 'DEFAULT';
	}

	static getById(db: DataSource, id: number | string) {
		try {
			return db.profile.findOne({
				id: typeof id === 'string' ? parseInt(id) : id,
			});
		} catch (e) {
			return null;
		}
	}

	static getForAccount(db: DataSource, acct: Account) {
		return db.profile.find({ active: true, accountId: acct.id });
	}

	static getAllShown(db: DataSource) {
		return db.profile.find({ active: true, visible: true });
	}

	static deselectAll(db: DataSource) {
		db.profile.update({ id: gt(0) as any }, { selected: false });
	}

	static hideProfile(db: DataSource, id: number) {
		db.profile.updateById(id, {
			visible: false,
		});
	}

	static unhideProfile(db: DataSource, id: number) {
		db.profile.updateById(id, {
			visible: true,
		});
	}

	static removeProfile(db: DataSource, id: number) {
		const match = db.profile.findOne({ id });
		//may never delete the default profile
		if (match && match.uuid === 'DEFAULT') return;
		db.profile.updateById(id, {
			active: false,
		});
	}

	static getDefaultProfile(db: DataSource, acct: Account) {
		const match = db.profile.findOne({
			accountId: acct.id,
			uuid: 'DEFAULT',
		});
		if (match) {
			Service._postSelect(db, match);
			return match;
		}
		return null;
	}

	static getActiveProfile(db: DataSource, acct: Account) {
		let found = db.profile.findOne({
			accountId: acct.id,
			selected: true,
		});
		if (!found) found = Service.getDefaultProfile(db, acct);
		if (!found)
			throw new Error(
				`No profile found for account ${acct.identifier} (${acct.server})`,
			);

		Service._postSelect(db, found);
		return found;
	}

	/**
	 * Makes sure the default profile is always populated
	 * @param db
	 * @param acct
	 */
	static setupDefaultProfile(db: DataSource, acct: Account) {
		let conflict = Service.getDefaultProfile(db, acct);
		if (!conflict) {
			db.profile.insert({
				accountId: acct.id,
				uuid: 'DEFAULT',
				name: 'Default',
				selected: true, // selected by default
			});
		} else {
			conflict = Service.getDefaultProfile(db, acct);
		}

		if (conflict) Service._postInsert(db, conflict);
	}

	static selectDefaultProfile(db: DataSource, acct: Account) {
		const conflict = Service.getDefaultProfile(db, acct);
		if (conflict)
			db.profile.update({ id: eq(conflict.id) as any }, { selected: true });
	}

	static getOwnerAccount(db: DataSource, profile: Profile): Account | null {
		return AccountService.getById(db, profile.accountId || -1);
	}

	/**
	 * Add a new profile for this account.
	 *
	 * name collisions are allowed
	 * @param db
	 * @param acct
	 * @param name
	 * @constructor
	 */
	static addProfile(db: DataSource, acct: Account, name: string) {
		const _key = RandomUtil.nanoId();
		db.profile.insert({
			accountId: acct.id,
			uuid: _key,
			name,
			selected: false,
		});

		const savedProfile = db.profile.findOne({
			uuid: _key,
		});

		if (savedProfile) Service._postInsert(db, savedProfile);
		return savedProfile;
	}

	/**
	 * Lifecycle Hooks
	 * @param db
	 * @param profile
	 */
	static _postInsert(db: DataSource, profile: Profile) {
		// upsert timeline items
		ProfilePinnedTimelineService.upsertDefaultTimelines(db, profile);
		ProfilePinnedTagService.upsertDefaultTags(db, profile);
	}

	static _postSelect(db: DataSource, profile: Profile) {
		// upsert timeline items
		ProfilePinnedTimelineService.upsertDefaultTimelines(db, profile);
		ProfilePinnedTagService.upsertDefaultTags(db, profile);
	}
}

export { Repo as ProfileRepo, Service as ProfileService };
