import { MastoAccountCredentials } from '#/types/mastojs.types.js';
import { MegaAccount } from '#/types/megalodon.types.js';

/**
 * Exchange a code for an access token
 */
export async function getMastodonAccessToken(
	instanceUrl: string,
	code: string,
	clientId: string,
	clientSecret: string,
) {
	const res = await fetch(`https://${instanceUrl}/oauth/token`, {
		method: 'POST',
		body: JSON.stringify({
			client_id: clientId,
			client_secret: clientSecret,
			redirect_uri: 'urn:ietf:wg:oauth:2.0:oob',
			grant_type: 'authorization_code',
			code,
			scope: 'read write push follow',
		}),
		headers: {
			'Content-Type': 'application/json',
		},
	});
	const data = await res.json();
	return data?.access_token;
}

export async function verifyCredentialsActivitypub(
	urlLike: string,
	token: string,
): Promise<MastoAccountCredentials | MegaAccount> {
	const res = await fetch(
		`https://${urlLike}/api/v1/accounts/verify_credentials`,
		{
			method: 'GET',
			headers: {
				Authorization: `Bearer ${token}`,
				'Content-Type': 'application/json',
			},
		},
	);
	return res.json();
}

/**
 *
 * @param urlLike
 * @param appName users will see this in their app list and timelines
 * @param appWebsite
 * @param callbackUrl
 * @param tokens make sure to cache these app credentials
 * (per instance specific, auto generated by Dhaaga, if not
 * present) in your local database. Repeated calls without
 * tokens will result in rate limiting.
 */
export async function generateActivitypubLoginUrl(
	urlLike: string,
	{ appName, appWebsite }: { appName: string; appWebsite: string },
	tokens: {
		appId?: string;
		appSecret?: string;
	},
) {
	let _appClientId = tokens?.appId;
	let _appClientSecret = tokens?.appSecret;

	/**
	 * If the client id and secret are not provided,
	 * we need to create one
	 */
	if (!_appClientId || !_appClientSecret) {
		const clientIdFormData: Record<string, string> = {
			client_name: appName,
			redirect_uris: 'urn:ietf:wg:oauth:2.0:oob',
			scopes: 'read write follow push',
			website: appWebsite,
		};

		// Generate the query string
		const clientIdQueryString = Object.keys(clientIdFormData)
			.map((key) => `${key}=${encodeURIComponent(clientIdFormData[key])}`)
			.join('&');

		const clientId = await fetch(`https://${urlLike}/api/v1/apps`, {
			method: 'POST',
			headers: {
				'Content-Type': 'application/x-www-form-urlencoded',
			},
			body: clientIdQueryString,
		});
		if (!clientId.ok) throw new Error('failed to generate app tokens ');

		const _body = await clientId.json();
		_appClientId = _body?.['client_id'];
		_appClientSecret = _body?.['client_secret'];
	}

	const authEndpoint = `https://${urlLike}/oauth/authorize`;

	// Set up parameters for the query string
	const options: Record<string, string> = {
		client_id: _appClientId!,
		redirect_uri: 'urn:ietf:wg:oauth:2.0:oob',
		response_type: 'code',
		scope: 'read write follow push',
	};
	// Generate the query string
	const queryString = Object.keys(options)
		.map((key) => `${key}=${encodeURIComponent(options[key])}`)
		.join('&');

	return {
		loginUrl: `${authEndpoint}?${queryString}`,
		// loginStrategy: 'code',
		// version: data.version,
		// software: data.software,
		clientId: _appClientId,
		clientSecret: _appClientSecret,
	};
}

export async function exchangeCodeForAccessToken(
	urlLike: string,
	code: string,
	clientId: string,
	clientSecret: string,
): Promise<string> {
	const res = await fetch(`https://${urlLike}/oauth/token`, {
		method: 'POST',
		body: JSON.stringify({
			client_id: clientId,
			client_secret: clientSecret,
			redirect_uri: 'urn:ietf:wg:oauth:2.0:oob',
			grant_type: 'authorization_code',
			code,
			scope: 'read write push follow',
		}),
		headers: {
			'Content-Type': 'application/json',
		},
	});
	const data = await res.json();
	return data?.access_token;
}
