import { TextParser } from './text.js';
import { RandomUtil } from '../utils/index.js';
import { DriverService } from '../services/driver.js';
export const WrapperNodes = [
    'para',
    'italic',
    'bold',
    'inline',
];
export class Builder {
    input;
    parsed;
    links;
    mentions;
    nodes;
    constructor({ input }) {
        this.input = input;
        this.parsed = [];
        this.links = new Map();
        this.mentions = [];
        this.nodes = [];
    }
    solve() {
        this.mentions = TextParser.findMentions(this.input);
        this.links = TextParser.findHyperlinks(this.input);
        this.nodes = TextParser.preprocessPostContent(this.input, false);
        this.process();
        return this;
    }
    process() {
        let count = 0;
        let paraCount = 0;
        for (const para of this.nodes) {
            this.parsed.push({
                uuid: RandomUtil.nanoId(),
                type: 'para',
                nodes: [],
            });
            for (const node of para) {
                if (node.type === 'text') {
                    const splits = node.props.text.split(/<br ?\/?>/);
                    this.parsed[paraCount].nodes.push({
                        uuid: RandomUtil.nanoId(),
                        nodes: [],
                        type: 'text',
                        text: splits[0],
                    });
                    count++;
                    for (let i = 1; i < splits.length; i++) {
                        this.parsed.push({
                            uuid: RandomUtil.nanoId(),
                            type: 'para',
                            nodes: [],
                        });
                        paraCount++;
                        this.parsed[paraCount].nodes.push({
                            uuid: RandomUtil.nanoId(),
                            type: 'text',
                            nodes: [],
                            text: splits[i],
                        });
                    }
                    const txt = node.props?.text.trim();
                    txt.replaceAll(/<br>/g, '\n');
                    continue;
                }
                const data = this.parser(node);
                if (data)
                    this.parsed[paraCount].nodes.push(data);
                count++;
            }
            paraCount++;
        }
    }
    parser(node) {
        try {
            switch (node.type) {
                case 'link':
                case 'url': {
                    const mention = this.mentions?.find((o) => o.url === node.props.url);
                    if (mention) {
                        return {
                            type: 'mention',
                            uuid: RandomUtil.nanoId(),
                            text: mention.text || null,
                            url: mention.url || null,
                            nodes: [],
                        };
                    }
                    const hashtag = TextParser.isHashtag(node.props.url);
                    if (hashtag)
                        return {
                            type: 'tag',
                            uuid: RandomUtil.nanoId(),
                            text: hashtag,
                            nodes: [],
                        };
                    let displayName = null;
                    if (this.links) {
                        const match = this.links.get(node.props.url);
                        if (match) {
                            displayName = match;
                        }
                    }
                    return {
                        type: 'link',
                        uuid: RandomUtil.nanoId(),
                        text: displayName || '',
                        url: node.props.url,
                        nodes: [],
                    };
                }
                case 'plain': {
                    return {
                        type: 'inline',
                        uuid: RandomUtil.nanoId(),
                        nodes: node.children
                            .map((o) => this.parser(o))
                            .filter(Boolean),
                    };
                }
                case 'italic': {
                    return {
                        type: 'italic',
                        uuid: RandomUtil.nanoId(),
                        nodes: node.children
                            .map((o) => this.parser(o))
                            .filter(Boolean),
                    };
                }
                case 'bold': {
                    return {
                        type: 'bold',
                        uuid: RandomUtil.nanoId(),
                        nodes: node.children
                            .map((o) => this.parser(o))
                            .filter(Boolean),
                    };
                }
                case 'mention': {
                    const mention = this.mentions?.find((o) => o.url === node.props.url);
                    return {
                        type: 'mention',
                        uuid: RandomUtil.nanoId(),
                        text: node.props.acct,
                        url: mention?.url || null,
                        nodes: [],
                    };
                }
                case 'inlineCode':
                    return {
                        type: 'code',
                        uuid: RandomUtil.nanoId(),
                        text: node.props.code,
                        nodes: [],
                    };
                case 'hashtag':
                    return {
                        type: 'tag',
                        text: node.props.hashtag,
                        uuid: RandomUtil.nanoId(),
                        nodes: [],
                    };
                case 'quote':
                case 'text':
                    return {
                        type: 'text',
                        text: node.props.text,
                        uuid: RandomUtil.nanoId(),
                        nodes: [],
                    };
                case 'emojiCode':
                    return {
                        type: 'customEmoji',
                        text: node.props.name,
                        value: node.props.name,
                        uuid: RandomUtil.nanoId(),
                        nodes: [],
                    };
                case 'unicodeEmoji':
                    return {
                        type: 'text',
                        text: node.props.emoji,
                        uuid: RandomUtil.nanoId(),
                        nodes: [],
                    };
                default: {
                    console.log('[WARN] [MFM]: node not evaluated', node);
                    return null;
                }
            }
        }
        catch (e) {
            console.log('[WARN] [MFM]: failed to process node', node.type, node.children, node.props, e);
            return null;
        }
    }
    decodeUrlString(input) {
        try {
            return decodeURI(input);
        }
        catch (e) {
            console.log('[ERROR]:', e, input);
            return input;
        }
    }
}
class Parser {
    static parse(driver, input, facets) {
        if (DriverService.supportsAtProto(driver))
            return Parser.withFacets(input, facets || []);
        return Parser.withMfm(input);
    }
    static toUtf8(input) {
        return new TextEncoder().encode(input);
    }
    static toUtf16(input) {
        return new TextDecoder('utf-8').decode(input);
    }
    static withMfm(input) {
        if (!input)
            return [];
        return new Builder({
            input,
        }).solve().parsed;
    }
    static withFacets(input, facets) {
        if (!input)
            return [];
        const byteArray = Parser.toUtf8(input);
        const elements = [];
        let idx = 0, count = 0;
        elements.push({
            uuid: RandomUtil.nanoId(),
            type: 'para',
            nodes: [],
        });
        for (const facet of facets) {
            const prefix = byteArray.slice(idx, facet.index.byteStart);
            elements[0].nodes.push({
                uuid: RandomUtil.nanoId(),
                nodes: [],
                type: 'text',
                text: Parser.toUtf16(prefix),
            });
            count++;
            const midSegment = byteArray.slice(facet.index.byteStart, facet.index.byteEnd);
            const _feature = facet.features[0];
            switch (_feature.$type) {
                case 'app.bsky.richtext.facet#mention': {
                    elements[0].nodes.push({
                        type: 'mention',
                        uuid: RandomUtil.nanoId(),
                        text: Parser.toUtf16(midSegment),
                        url: _feature.did,
                        nodes: [],
                    });
                    break;
                }
                case 'app.bsky.richtext.facet#link': {
                    elements[0].nodes.push({
                        type: 'link',
                        uuid: RandomUtil.nanoId(),
                        text: Parser.toUtf16(midSegment),
                        url: _feature.uri,
                        nodes: [],
                    });
                    break;
                }
                case 'app.bsky.richtext.facet#tag': {
                    elements[0].nodes.push({
                        type: 'tag',
                        uuid: RandomUtil.nanoId(),
                        text: Parser.toUtf16(midSegment),
                        nodes: [],
                    });
                    break;
                }
            }
            count++;
            idx = facet.index.byteEnd;
        }
        const suffix = byteArray.slice(idx);
        elements[0].nodes.push({
            uuid: RandomUtil.nanoId(),
            nodes: [],
            type: 'text',
            text: Parser.toUtf16(suffix),
        });
        return elements;
    }
}
export { Parser as TextNodeParser };
//# sourceMappingURL=text-nodes.js.map