import { UserParser } from './user.js';
import { TextNodeParser } from './text-nodes.js';
import { DriverService } from '../services/driver.js';
import { RandomUtil } from '../utils/index.js';
import { ActivitypubStatusAdapter } from '../implementors/status/_adapters.js';
import { ActivitypubHelper } from '../index.js';
import { postObjectSchema, } from '#/types/shared/post.js';
class Parser {
    static _export(input, domain, subdomain) {
        if (!input)
            return null;
        const IS_BOOKMARK_RESOLVED = DriverService.supportsMastoApiV1(domain);
        const IS_ATPROTO = DriverService.supportsAtProto(domain);
        const medias = input?.getMediaAttachments();
        const user = UserParser.rawToInterface(input.getUser(), domain);
        let handle = IS_ATPROTO
            ? `@${user.getUsername()}`
            : ActivitypubHelper.getHandle(input.getAccountUrl(subdomain) || '', subdomain);
        const parsedContent = TextNodeParser.parse(domain, input.getContent() || '', input.getFacets());
        const parsedDisplayName = IS_ATPROTO
            ? [
                {
                    uuid: RandomUtil.nanoId(),
                    type: 'para',
                    nodes: [
                        {
                            uuid: RandomUtil.nanoId(),
                            type: 'text',
                            text: user.getDisplayName(),
                        },
                    ],
                },
            ]
            : TextNodeParser.parse(domain, input.getDisplayName() || '', input.getFacets());
        return {
            uuid: input.getId(),
            id: input.getId(),
            visibility: input.getVisibility(),
            createdAt: input.getCreatedAt(),
            postedBy: {
                id: user.getId(),
                avatarUrl: user.getAvatarUrl() || '',
                displayName: user.getDisplayName(),
                parsedDisplayName: parsedDisplayName || [],
                handle: handle,
                instance: user.getInstanceUrl() || subdomain,
            },
            content: {
                raw: input.getContent(),
                parsed: parsedContent || [],
                media: medias.map((o) => ({
                    height: o.getHeight(),
                    width: o.getWidth(),
                    alt: o.getAltText() || null,
                    blurhash: o.getBlurHash() || null,
                    type: o.getType(),
                    url: o.getUrl(),
                    previewUrl: o.getPreviewUrl(),
                })) || [],
                links: input.getLinkAttachments(),
            },
            stats: {
                replyCount: input.getRepliesCount(),
                boostCount: input.getRepostsCount(),
                likeCount: input.getFavouritesCount(),
                quoteCount: input.getQuotesCount(),
                reactions: input.getReactions(input.getMyReaction() || ''),
            },
            interaction: {
                bookmarked: input.getIsBookmarked() || false,
                boosted: input.getIsRebloggedByMe() || false,
                liked: input.getIsFavourited() || false,
            },
            calculated: {
                emojis: new Map([...user.getEmojiMap(), ...input.getCachedEmojis()]),
                mediaContainerHeight: 0, // height,
                reactionEmojis: input.getReactionEmojis(),
                mentions: input.getMentions(),
            },
            meta: {
                sensitive: input.getIsSensitive(),
                cw: input.getSpoilerText() || null,
                isBoost: input.isReposted(),
                isReply: input.isReply(),
                mentions: input.getMentions(),
                cid: input.getCid(),
                uri: input.getUri(),
            },
            state: {
                isBookmarkStateFinal: IS_BOOKMARK_RESOLVED,
            },
            atProto: {
                viewer: DriverService.supportsAtProto(domain)
                    ? input.getViewer()
                    : undefined,
            },
        };
    }
    static rawToInterface(input, driver) {
        if (Array.isArray(input)) {
            return input
                .filter((o) => !!o)
                .map((o) => ActivitypubStatusAdapter(o, driver));
        }
        else {
            return ActivitypubStatusAdapter(input, driver);
        }
    }
    static interfaceToJson(input, { driver, server, }) {
        if (!input)
            return null;
        const IS_SHARE = input.isReposted();
        const HAS_PARENT = input.isReply();
        const HAS_ROOT = input.hasRootAvailable();
        let sharedFrom = IS_SHARE
            ? Parser.parse(input.getRepostedStatusRaw(), driver, server)
            : null;
        let replyTo = HAS_PARENT
            ? Parser.parse(input.getParentRaw(), driver, server)
            : null;
        let root = HAS_ROOT
            ? Parser.parse(input.getRootRaw(), driver, server)
            : null;
        const dto = HAS_PARENT &&
            (DriverService.supportsAtProto(driver) ||
                DriverService.supportsMisskeyApi(driver))
            ?
                {
                    ...Parser._export(input, driver, server),
                    boostedFrom: sharedFrom,
                    replyTo,
                    rootPost: root,
                }
            : {
                ...Parser._export(input, driver, server),
                boostedFrom: sharedFrom,
            };
        const { data, error, success } = postObjectSchema.safeParse(dto);
        if (!success) {
            console.log('[ERROR]: status item dto validation failed', error);
            return null;
        }
        return data;
    }
    static parse(input, driver, server) {
        if (input instanceof Array) {
            return input
                .map((o) => Parser.rawToInterface(o, driver))
                .filter((o) => !!o)
                .map((o) => Parser.interfaceToJson(o, {
                driver,
                server,
            }))
                .filter((o) => !!o);
        }
        else {
            try {
                if (!input)
                    return null;
                return Parser.interfaceToJson(Parser.rawToInterface(input, driver), {
                    driver,
                    server,
                });
            }
            catch (e) {
                console.log('[ERROR]: failed to deserialize post object', e, 'input:', input, driver, server);
                return null;
            }
        }
    }
}
class Inspector {
    static getContentTarget(input) {
        if (!input) {
            console.log('[WARN]: trying to obtain target post for', input);
            return input;
        }
        if (input.meta.isBoost && !input.boostedFrom) {
            console.log('[WARN]: original object not available for a repost', input);
            return input;
        }
        return input.meta.isBoost
            ? input.content.raw || input.content.media.length > 0
                ? input
                : input.boostedFrom
            : input;
    }
    static isQuoteObject(input) {
        return (input?.meta?.isBoost &&
            (input?.content?.raw || input?.content?.media?.length > 0));
    }
    static isLiked(input) {
        if (!input)
            return false;
        return !!input.atProto?.viewer?.like || input.interaction.liked;
    }
    static isShared(input) {
        if (!input)
            return false;
        return !!input.atProto?.viewer?.repost || input.interaction.boosted;
    }
    static isBookmarked(input) {
        const _target = Inspector.getContentTarget(input);
        return _target.interaction.bookmarked;
    }
}
class Resolver {
    static mentionItemsToWebfinger(handle, items) {
        const parts = handle.split('@').filter(Boolean);
        if (parts.length === 1) {
            const match = items.find((o) => o.acct?.startsWith(parts[0]) && o.url?.endsWith(parts[0]));
            if (match) {
                return {
                    use: 'userId',
                    userId: match.id,
                };
            }
        }
        const match = items.find((o) => o.acct === `${parts[0]}@${parts[1]}`);
        if (match) {
            return {
                use: 'userId',
                userId: match.id,
            };
        }
        return null;
    }
}
export { Parser as PostParser, Inspector as PostInspector, Resolver as PostResolver, };
//# sourceMappingURL=post.js.map