import { z } from 'zod';
import { UserParser } from '#/parsers/index.js';
import { DriverService } from '#/services/driver.js';
import { ApiErrorCode } from '#/types/api-response.js';
const DriverWebfingerSchema = z.object({
    username: z.string(),
    host: z.string().nullable(),
});
const userFindQuerySchema = z
    .object({
    use: z.literal('did'),
    did: z.string(),
})
    .or(z.object({
    use: z.literal('userId'),
    userId: z.string(),
}))
    .or(z.object({
    use: z.literal('handle'),
    handle: z.string(),
}))
    .or(z.object({
    use: z.literal('webfinger'),
    webfinger: DriverWebfingerSchema,
}));
export { userFindQuerySchema };
export async function unifiedUserLookup(client, query) {
    const driver = client.driver;
    const server = client.server;
    switch (query.use) {
        case 'did': {
            const data = await client.users.get(query.did);
            return UserParser.parse(data.data, driver, server);
        }
        case 'userId': {
            const data = await client.users.get(query.userId);
            return UserParser.parse(data, driver, server);
        }
        case 'handle': {
            if (DriverService.supportsAtProto(driver)) {
                const didData = await client.users.getDid(query.handle);
                const data = await client.users.get(didData?.data?.did);
                return UserParser.parse(data.data, driver, server);
            }
            else if (DriverService.supportsMastoApiV1(client.driver)) {
                const data = await client.users.lookup({
                    username: query.handle,
                    host: null,
                });
                return UserParser.parse(data, driver, server);
            }
            else if (DriverService.supportsMisskeyApi(client.driver)) {
                throw new Error(ApiErrorCode.OPERATION_UNSUPPORTED);
            }
            else {
                throw new Error(ApiErrorCode.OPERATION_UNSUPPORTED);
            }
        }
        case 'webfinger': {
            if (DriverService.supportsMastoApiV1(client.driver)) {
                const data = await client.users.lookup(query.webfinger);
                return UserParser.parse(data, driver, server);
            }
            else if (DriverService.supportsMisskeyApi(client.driver)) {
                const findResult = await client.users.findByWebfinger(query.webfinger);
                return UserParser.parse(findResult.data, driver, server);
            }
            else {
                throw new Error(ApiErrorCode.OPERATION_UNSUPPORTED);
            }
        }
        default: {
            throw new Error(ApiErrorCode.OPERATION_UNSUPPORTED);
        }
    }
}
//# sourceMappingURL=unifiedUserLookup.js.map