import { CasingUtil } from '#/utils/casing.js';
import { BaseUrlNormalizationService } from '#/utils/urls.js';
import { getHumanReadableError } from '#/utils/errors.js';
class FetchWrapper {
    baseUrl;
    token;
    requestHeader;
    constructor(urlLike, token) {
        this.baseUrl = BaseUrlNormalizationService.appendHttps(urlLike);
        this.token = token;
        this.requestHeader = this.token
            ? {
                'Content-Type': 'application/json',
                Authorization: `Bearer ${this.token}`,
            }
            : {
                'Content-Type': 'application/json',
            };
        return this;
    }
    static create(urlLike, token) {
        return new FetchWrapper(urlLike, token);
    }
    static cleanObject(obj) {
        Object.keys(obj).forEach((key) => {
            if (obj[key] === null || obj[key] === undefined) {
                delete obj[key];
            }
        });
        let typesOverride = obj['types[]'];
        const retval = CasingUtil.snakeCaseKeys(obj);
        if (typesOverride) {
            delete retval['types'];
            retval['types[]'] = typesOverride;
        }
        return retval;
    }
    withQuery(endpoint, query) {
        if (!query)
            return `${this.baseUrl}${endpoint}`;
        if (query['types[]'] !== undefined) {
            const sample = FetchWrapper.cleanObject(query);
            const items = sample['types[]'].split(';');
            delete sample['query[]'];
            const params = new URLSearchParams(FetchWrapper.cleanObject(sample));
            for (const item of items) {
                params.append('types[]', item);
            }
            return `${this.baseUrl}${endpoint}?` + params.toString();
        }
        return (`${this.baseUrl}${endpoint}?` +
            new URLSearchParams(FetchWrapper.cleanObject(query)));
    }
    async getCamelCaseWithLinkPagination(endpoint, query) {
        const url = this.withQuery(endpoint, query);
        const response = await fetch(url, {
            method: 'GET',
            headers: this.requestHeader,
        });
        if (!response.ok) {
            throw new Error(response.statusText ?? String(response.status));
        }
        const { minId, maxId } = DhaagaApiUtils.extractPaginationFromLinkHeader(response.headers);
        const json = await response.json();
        const data = CasingUtil.camelCaseKeys(json);
        return { data, minId, maxId };
    }
    async getCamelCase(endpoint, query) {
        const url = this.withQuery(endpoint, query);
        const response = await fetch(url, {
            method: 'GET',
            headers: this.requestHeader,
        });
        if (!response.ok)
            throw new Error(response.statusText ?? response.status);
        const json = await response.json();
        return CasingUtil.camelCaseKeys(json);
    }
    async post(endpoint, body, opts) {
        const url = `${this.baseUrl}${endpoint}`;
        const headers = {
            'Content-Type': 'application/json',
            ...(this.token && { Authorization: `Bearer ${this.token}` }),
        };
        const response = await fetch(url, {
            method: 'POST',
            headers,
            body: JSON.stringify(body),
            ...opts,
        });
        if (!response.ok)
            throw new Error(response.statusText ?? response.status);
        return response.json();
    }
    static applyQueriesToRequestUrl(url, opts) {
        return opts.queries
            ? `${opts.baseURL}${url}?` +
                new URLSearchParams(FetchWrapper.cleanObject(opts.queries))
            : `${opts.baseURL}${url}?`;
    }
    static buildRequestInitObject(opts) {
        if (!opts)
            return {};
        return {
            ...opts,
            headers: opts.accessToken
                ? {
                    'Content-Type': 'application/json',
                    Authorization: `Bearer ${opts.accessToken}`,
                }
                : {
                    'Content-Type': 'application/json',
                },
        };
    }
    async get(endpoint, opts) {
        const url = FetchWrapper.applyQueriesToRequestUrl(endpoint, {
            ...opts,
            baseURL: opts?.baseURL || this.baseUrl,
        });
        try {
            const response = await fetch(url, FetchWrapper.buildRequestInitObject({ ...opts, method: 'GET' }));
            if (!response.ok) {
                let message = response.statusText;
                let errorCode = null;
                let errorBody;
                try {
                    errorBody = await response.json();
                    if (typeof errorBody?.error === 'string') {
                        message = errorBody.error;
                    }
                    else if (typeof errorBody?.error?.message === 'string') {
                        message = errorBody.error.message;
                    }
                }
                catch {
                    errorBody = await response.text();
                    console.log('via body', errorBody);
                }
                switch (response.status) {
                    case 401:
                        errorCode = 'unauthorized';
                        throw new Error('unauthorized');
                }
            }
            let data = await response.json();
            if (opts?.transformResponse === 'snake') {
                data = CasingUtil.snakeCaseKeys(data);
            }
            else if (opts?.transformResponse === 'camel') {
                data = CasingUtil.camelCaseKeys(data);
            }
            return data;
        }
        catch (err) {
            throw new Error(getHumanReadableError(err));
        }
    }
}
class DhaagaApiUtils {
    static extractPaginationFromLinkHeader(headers) {
        const linkHeader = headers?.map?.link;
        const maxIdRegex = /max_id=([0-9]+)/;
        const minIdRegex = /min_id=([0-9]+)/;
        let maxId = null;
        let minId = null;
        if (minIdRegex.test(linkHeader)) {
            const minMatch = linkHeader.match(minIdRegex);
            minId = minMatch[1];
        }
        if (maxIdRegex.test(linkHeader)) {
            const maxMatch = linkHeader.match(maxIdRegex);
            maxId = maxMatch[1];
        }
        return { minId, maxId };
    }
}
export default FetchWrapper;
//# sourceMappingURL=fetch.js.map