import { camelCase, snakeCase } from "change-case";
import { MastoDeserializeError, MastoUnexpectedError, } from "../errors/index.js";
import { flattenForFormData, flattenForRailsQueryString } from "./flatten.js";
import { transformKeys } from "./transform-keys.js";
export class SerializerNativeImpl {
    serialize(type, rawData) {
        const data = transformKeys(rawData, snakeCase);
        switch (type) {
            case "json": {
                return JSON.stringify(data);
            }
            case "multipart-form": {
                const formData = new FormData();
                for (const [key, value] of Object.entries(flattenForFormData(data))) {
                    formData.append(key, value);
                }
                return formData;
            }
            case "querystring": {
                return flattenForRailsQueryString(data);
            }
            default: {
                throw new MastoUnexpectedError(`Unknown content type ${type} to serialize.`);
            }
        }
    }
    deserialize(type, data) {
        switch (type) {
            case "json": {
                try {
                    return transformKeys(JSON.parse(data), camelCase);
                }
                catch {
                    throw new MastoDeserializeError(`Malformed JSON ${data} returned from the server.`, type, data);
                }
            }
            default: {
                throw new MastoDeserializeError(`Unknown content type ${type} returned from the server.`, type, data);
            }
        }
    }
}
