"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fromBase64Node = exports.fromBase64Native = void 0;
exports.fromBase64Ponyfill = fromBase64Ponyfill;
const from_string_1 = require("uint8arrays/from-string");
const nodejs_buffer_js_1 = require("./lib/nodejs-buffer.js");
const Buffer = nodejs_buffer_js_1.NodeJSBuffer;
exports.fromBase64Native = typeof Uint8Array.fromBase64 === 'function'
    ? function fromBase64Native(b64, alphabet = 'base64') {
        return Uint8Array.fromBase64(b64, {
            alphabet,
            lastChunkHandling: 'loose',
        });
    }
    : null;
exports.fromBase64Node = Buffer
    ? function fromBase64Node(b64, alphabet = 'base64') {
        const bytes = Buffer.from(b64, alphabet);
        verifyBase64ForBytes(b64, bytes);
        // Convert to Uint8Array because even though Buffer is a sub class of
        // Uint8Array, it serializes differently to Uint8Array (e.g. in JSON) and
        // results in unexpected behavior downstream (e.g. in tests)
        return new Uint8Array(bytes.buffer, bytes.byteOffset, bytes.byteLength);
    }
    : null;
function fromBase64Ponyfill(b64, alphabet = 'base64') {
    const bytes = (0, from_string_1.fromString)(b64, b64.endsWith('=') ? `${alphabet}pad` : alphabet);
    verifyBase64ForBytes(b64, bytes);
    return bytes;
}
// @NOTE NodeJS will silently stop decoding at the first invalid character,
// while "uint8arrays/from-string" will not validate that the padding is
// correct. The following function performs basic validation to ensure that the
// input was a valid base64 string. The availability of the "bytes" allows
// to perform checks with O[1] complexity.
function verifyBase64ForBytes(b64, bytes) {
    const paddingCount = b64.endsWith('==') ? 2 : b64.endsWith('=') ? 1 : 0;
    const trimmedLength = b64.length - paddingCount;
    const expectedByteLength = Math.floor((trimmedLength * 3) / 4);
    if (bytes.length !== expectedByteLength) {
        throw new Error('Invalid base64 string');
    }
    const expectedB64Length = (bytes.length / 3) * 4;
    const expectedPaddingCount = expectedB64Length % 4 === 0 ? 0 : 4 - (expectedB64Length % 4);
    const expectedFullB64Length = expectedB64Length + expectedPaddingCount;
    if (b64.length > expectedFullB64Length) {
        throw new Error('Invalid base64 string');
    }
    // The previous might still allow false positive if only the last few
    // chars are invalid.
    for (let i = Math.ceil(expectedB64Length); i < b64.length - paddingCount; i++) {
        const code = b64.charCodeAt(i);
        if (!(code >= 65 && code <= 90) && // A-Z
            !(code >= 97 && code <= 122) && // a-z
            !(code >= 48 && code <= 57) && // 0-9
            code !== 43 && // +
            code !== 47 // /
        ) {
            throw new Error('Invalid base64 string');
        }
    }
}
//# sourceMappingURL=uint8array-from-base64.js.map