import { queryOptions } from '@tanstack/react-query';
import { ActivitypubHelper, ActivityPubService, DriverService, PostParser, UserParser, } from '@dhaaga/bridge';
import { unifiedUserLookup } from '@dhaaga/bridge';
export function userProfileQueryOpts(client, query) {
    async function api() {
        if (query.use === 'handle') {
            const webfinger = ActivitypubHelper.splitHandle(query.handle, client.server);
            if (!webfinger)
                throw new Error(`failed to resolve webfinger from handle -> ${query.handle}`);
            query = { use: 'webfinger', webfinger };
        }
        return unifiedUserLookup(client, query);
    }
    return queryOptions({
        queryKey: [client.key, 'dhaaga/user', query],
        queryFn: api,
        enabled: !!client && !!query,
    });
}
export function userFollowsQueryOpts(client, userId, maxId) {
    async function api() {
        const data = await client.users.getFollowings({
            id: userId,
            limit: 15,
            maxId,
            allowPartial: true,
        });
        return {
            ...data,
            data: UserParser.parse(data.data, client.driver, client.server),
        };
    }
    return queryOptions({
        queryKey: ['dhaaga/user/follows', client.key, userId, maxId],
        queryFn: api,
        enabled: !!client,
    });
}
export function userFollowersQueryOpts(client, acctId, maxId) {
    async function api() {
        const data = await client.users.getFollowers({
            id: acctId,
            limit: 10,
            maxId,
            allowPartial: true,
        });
        return {
            ...data,
            data: UserParser.parse(data.data, client.driver, client.server),
        };
    }
    return queryOptions({
        queryKey: [client.key, 'dhaaga/user/followers', acctId, maxId],
        queryFn: api,
        enabled: !!client,
    });
}
export function userGalleryQueryOpts(client, userId) {
    async function api(client, userId) {
        const result = await client.users.getPosts(userId, {
            limit: 10,
            userId,
            onlyMedia: true,
            excludeReblogs: true,
            allowPartial: true,
            withFiles: true,
            withRenotes: false,
            withReplies: false,
            bskyFilter: DriverService.supportsAtProto(client.driver)
                ? 'posts_with_media'
                : undefined,
        });
        const data = DriverService.supportsAtProto(client.driver)
            ? PostParser.parse(result.data.map((o) => o.post), client.driver, client.server)
            : PostParser.parse(result.data, client.driver, client.server);
        return {
            data,
        };
    }
    return queryOptions({
        queryKey: [`dhaaga/profile/gallery`, userId],
        queryFn: () => api(client, userId),
        enabled: !!client && !!userId,
    });
}
export function userGetPinnedPosts(client, userId) {
    async function api() {
        if (ActivityPubService.misskeyLike(client.driver)) {
            const data = await client.users.get(userId);
            const _data = data;
            return PostParser.parse(_data.pinnedNotes, client.driver, client.server).slice(0, 10);
        }
        else if (ActivityPubService.blueskyLike(client.driver)) {
            return PostParser.parse(await client.users.getPinnedPosts(userId), client.driver, client.server).slice(0, 10);
        }
        else {
            const data = await client.users.getPosts(userId, {
                limit: 10,
                pinned: true,
                userId,
            });
            return PostParser.parse(data, client.driver, client.server).slice(0, 10);
        }
    }
    return queryOptions({
        queryKey: ['dhaaga/user/pins', client.key, userId],
        queryFn: api,
        enabled: !!client && !!userId,
    });
}
//# sourceMappingURL=user.js.map