import { PostParser } from '#/parsers/post.js';
import { UserParser } from '#/parsers/user.js';
import { produce } from 'immer';
import { RandomUtil } from '#/utils/index.js';
class Parser {
    static parseForMastodonV1(input, driver, server, category) {
        return {
            data: input.data.map((o) => {
                return {
                    id: o.id,
                    type: category === 'updates' ? 'status' : o.type,
                    post: PostParser.parse(o.status, driver, server),
                    user: UserParser.parse(o.account, driver, server),
                    read: o.pleroma?.isSeen, // also have o.pleroma.isMuted
                    createdAt: new Date(o.createdAt),
                    extraData: {},
                };
            }),
            maxId: input.maxId,
            minId: input.minId,
        };
    }
    static parseForMastodonV2(input, driver, server) {
        const acctList = input.data.accounts;
        const postList = input.data.statuses;
        const seenPost = new Map();
        let counter = 0;
        const results = [];
        for (const group of input.data.notificationGroups) {
            const _group = group;
            if (_group.statusId === undefined) {
                results.push({
                    id: group.groupKey,
                    type: group.type,
                    user: null,
                    post: null,
                    users: group.sampleAccountIds.map((o) => ({
                        item: UserParser.parse(acctList.find((x) => x.id === o), driver, server),
                        types: [group.type],
                        extraData: {},
                    })),
                    read: true,
                    createdAt: group.latestPageNotificationAt
                        ? new Date(group.latestPageNotificationAt)
                        : new Date(),
                    extraData: {},
                });
                counter++;
            }
            else if (!seenPost.has(group.statusId)) {
                const post = PostParser.parse(postList.find((x) => x.id === group.statusId), driver, server);
                results.push({
                    id: group.groupKey,
                    type: group.type,
                    user: null,
                    post,
                    users: group.sampleAccountIds.map((o) => ({
                        item: UserParser.parse(acctList.find((x) => x.id === o), driver, server),
                        types: [group.type],
                        extraData: {},
                    })),
                    read: true,
                    createdAt: group.latestPageNotificationAt
                        ? new Date(group.latestPageNotificationAt)
                        : new Date(),
                    extraData: {
                        interaction: post.interaction,
                    },
                });
                seenPost.set(group.statusId, counter);
                counter++;
            }
            else {
                const idx = seenPost.get(group.statusId);
                for (const id of group.sampleAccountIds) {
                    results[idx] = produce(results[idx], (draft) => {
                        const match = draft.users?.find((o) => o.item.id === id);
                        if (match) {
                            if (!match.types.includes(group.type)) {
                                match.types.push(group.type);
                            }
                        }
                        else {
                            draft.users?.push({
                                item: {
                                    ...UserParser.parse(acctList.find((x) => x.id === id), driver, server),
                                },
                                extraData: {},
                                types: [group.type],
                            });
                        }
                    });
                }
            }
        }
        return {
            data: results,
            maxId: input.maxId,
            minId: input.minId,
        };
    }
    static parseForMisskey(data, driver, server) {
        return {
            data: data.data
                .map((o) => {
                try {
                    if (['achievementEarned', 'note:grouped'].includes(o.type)) {
                        return null;
                    }
                    const _postTarget = !!o.note ? o.note : o;
                    const _acct = !['login'].includes(o.type)
                        ? UserParser.parse(o.user, driver, server)
                        : null;
                    const _post = _postTarget &&
                        !['login', 'follow', 'followRequestAccepted'].includes(o.type)
                        ? PostParser.parse(_postTarget, driver, server)
                        : null;
                    if (!_post) {
                        console.log('[WARN]: failed to pack notification for', o);
                        return null;
                    }
                    const _obj = {
                        id: RandomUtil.nanoId(),
                        type: o.type ||
                            (_post.visibility === 'specified' ? 'chat' : _post.visibility),
                        post: _post,
                        user: _acct,
                        read: false,
                        createdAt: new Date(o.createdAt || _post.createdAt),
                        extraData: {},
                    };
                    return _obj;
                }
                catch (e) {
                    console.log('[WARN]: failed to resolve notification', e);
                    return null;
                }
            })
                .filter(Boolean),
            minId: null,
            maxId: data.data[data.data.length - 1].id,
        };
    }
    static async parseForBluesky(data, client, driver, server) {
        let results = data.data.map((o) => {
            const _user = UserParser.parse(o.author, driver, server);
            const _post = o.record && o.record.$type === 'app.bsky.feed.post'
                ? PostParser.parse(o, driver, server)
                : null;
            return {
                id: o.uri,
                uri: o.uri,
                cid: o.cid,
                type: o.reason,
                user: _user,
                post: _post,
                createdAt: new Date(o.indexedAt),
                read: o.isRead,
                extraData: {},
                users: [],
            };
        });
        const uris = results.map((o) => o.uri);
        const posts = await client.posts.getPosts(uris);
        const parsed = PostParser.parse(posts, driver, server);
        results = results.map((o) => ({
            ...o,
            post: parsed.find((x) => x.id === o.uri) || o.post,
        }));
        return {
            ...data,
            data: results,
        };
    }
}
export { Parser as GroupedNotificationParser };
//# sourceMappingURL=notification-grouped.js.map