import { z } from 'zod';
import { RandomUtil } from '#/utils/random.js';
class BlueskyMediaAttachmentAdapter {
    item;
    constructor(ref) {
        this.item = ref;
    }
    static create(ref) {
        return new BlueskyMediaAttachmentAdapter(ref);
    }
    getId() {
        const ex = /^https:\/\/.*?\/did:plc:.*?\/(.*?)@.*?$/;
        if (this.item?.fullsize && ex.test(this.item.fullsize)) {
            return this.item.fullsize.match(ex)[1];
        }
        return '';
    }
    getAltText = () => this.item.alt;
    getBlurHash = () => null;
    getCreatedAt = () => new Date().toString();
    getMeta = () => { };
    getName = () => '';
    getPreviewUrl = () => this.item.thumb;
    getUrl = () => this.item.fullsize;
    getHeight = () => this.item?.aspectRatio?.height;
    getWidth = () => this.item?.aspectRatio?.width;
    getType() {
        return 'image';
    }
    print() {
        console.log(this.item);
    }
}
class BlueskyVideoAttachmentAdapter {
    item;
    constructor(ref) {
        this.item = ref;
    }
    static create(ref) {
        return new BlueskyVideoAttachmentAdapter(ref);
    }
    getId() {
        return this.item.cid;
    }
    getAltText = () => null;
    getBlurHash = () => null;
    getCreatedAt = () => new Date().toString();
    getMeta = () => { };
    getName = () => '';
    getPreviewUrl = () => this.item.thumbnail;
    getUrl = () => {
        return this.item.thumbnail
            .replace('video.bsky.app/watch', 'video.cdn.bsky.app/hls')
            .replace('thumbnail.jpg', '360p/video0.ts');
    };
    getHeight = () => this.item?.aspectRatio?.height;
    getWidth = () => this.item?.aspectRatio?.width;
    getType() {
        return 'video';
    }
    print() {
        console.log(this.item);
    }
}
export const bskyEmbedExternalSchema_2 = z.object({
    $type: z.literal('app.bsky.embed.external#view'),
    external: z.object({
        uri: z.string(),
        title: z.string(),
        description: z.string(),
        thumb: z.string(),
    }),
});
class EmbedViewProcessor_External_2 {
    item;
    constructor(obj) {
        this.item = obj;
    }
    static create(obj) {
        return new EmbedViewProcessor_External_2(obj);
    }
    static isCompatible(obj) {
        const { success } = bskyEmbedExternalSchema.safeParse(obj);
        return success;
    }
    static compile(obj) {
        const { data } = bskyEmbedExternalSchema.safeParse(obj);
        return [EmbedViewProcessor_External_2.create(data)];
    }
    getId() {
        return RandomUtil.nanoId();
    }
    getAltText = () => this.item.external.description;
    getBlurHash = () => null;
    getCreatedAt = () => new Date().toString();
    getMeta = () => { };
    getName = () => '';
    getPreviewUrl = () => this.item.external.thumb;
    getUrl = () => this.item.external.uri;
    getHeight = () => null;
    getWidth = () => null;
    getType() {
        return 'image/gif';
    }
    print() {
        console.log(this.item);
    }
}
const bskyEmbedImagesSchemaImageItem = z.object({
    alt: z.string(), // can be empty string
    aspectRatio: z.object({
        height: z.number().int(),
        width: z.number().int(),
    }),
    thumb: z.string(),
    fullsize: z.string(),
});
const bskyEmbedImagesViewSchema = z.object({
    $type: z.literal('app.bsky.embed.images#view'),
    images: z.array(bskyEmbedImagesSchemaImageItem),
});
const bskyEmbedImageSchema = z.object({
    $type: z.literal('app.bsky.embed.images'),
    images: z.array(z.object({
        alt: z.string(),
        aspectRatio: z.object({
            height: z.number().int(),
            width: z.number().int(),
        }),
        image: z.object({
            $type: z.literal('blob'),
            ref: z.object({
                $link: z.string(),
            }),
            mimeType: z.string(),
            size: z.number().int(),
        }),
    })),
});
class EmbedViewProcessor_Images {
    static isCompatible(obj) {
        const { success } = bskyEmbedImageSchema.safeParse(obj);
        return success;
    }
    static compile(obj) {
        const { data } = bskyEmbedImageSchema.safeParse(obj);
        return data.images.map((o) => BlueskyMediaAttachmentAdapter.create({
            alt: o.alt,
            aspectRatio: o.aspectRatio,
            fullsize: o.image.ref.$link,
            thumb: o.image.ref.$link,
        }));
    }
}
class EmbedViewProcessor_ImagesView {
    static isCompatible(obj) {
        const { success } = bskyEmbedImagesViewSchema.safeParse(obj);
        return success;
    }
    static compile(obj) {
        const { data } = bskyEmbedImagesViewSchema.safeParse(obj);
        return data.images.map((o) => BlueskyMediaAttachmentAdapter.create(o));
    }
}
const bskyEmbedVideoSchema = z.object({
    $type: z.literal('app.bsky.embed.video#view'),
    cid: z.string(),
    playlist: z.string(),
    thumbnail: z.string(),
    aspectRatio: z.object({
        height: z.number().int(),
        width: z.number().int(),
    }),
});
class EmbedViewProcessor_Video {
    static isCompatible(obj) {
        const { success } = bskyEmbedVideoSchema.safeParse(obj);
        return success;
    }
    static compile(obj) {
        const { data } = bskyEmbedVideoSchema.safeParse(obj);
        return [BlueskyVideoAttachmentAdapter.create(data)];
    }
}
const bskyEmbedExternalSchema = z.object({
    $type: z.literal('app.bsky.embed.external#view'),
    external: z.object({
        uri: z.string(),
        title: z.string(),
        description: z.string(),
        thumb: z.string(),
    }),
});
class LinkEmbedProcessor_External {
    static isCompatible(obj) {
        const { success } = bskyEmbedExternalSchema.safeParse(obj);
        return success;
    }
    static compile(obj) {
        const { data } = bskyEmbedExternalSchema.safeParse(obj);
        return [
            {
                ...data.external,
                url: data.external.uri,
                bannerImageUrl: data.external.thumb,
            },
        ];
    }
}
const bskyEmbedRecordWithMediaSchema = z.object({
    $type: z.literal('app.bsky.embed.recordWithMedia#view'),
    media: z.any(),
    record: z.any(),
});
class EmbedViewProcessor_RecordWithMedia {
    static isCompatible(obj) {
        const { success } = bskyEmbedRecordWithMediaSchema.safeParse(obj);
        return success;
    }
    static compile(obj) {
        const { data } = bskyEmbedRecordWithMediaSchema.safeParse(obj);
        if (EmbedViewProcessor_ImagesView.isCompatible(data.media))
            return EmbedViewProcessor_ImagesView.compile(data.media);
        if (EmbedViewProcessor_Video.isCompatible(data.media))
            return EmbedViewProcessor_Video.compile(data.media);
        return [];
    }
}
export { EmbedViewProcessor_External_2, LinkEmbedProcessor_External, EmbedViewProcessor_ImagesView, EmbedViewProcessor_RecordWithMedia, EmbedViewProcessor_Video, BlueskyMediaAttachmentAdapter, BlueskyVideoAttachmentAdapter, };
//# sourceMappingURL=bluesky.js.map