import FetchWrapper from '#/client/utils/fetch.js';
import { MastoJsWrapper, MegalodonGoToSocialWrapper, MisskeyJsWrapper, } from '#/client/utils/api-wrappers.js';
import { KNOWN_SOFTWARE } from '#/client/utils/driver.js';
import { identifyBackendSoftware } from '#/client/utils/detect-software.js';
import { errorBuilder } from '#/types/index.js';
import { ApiErrorCode } from '#/types/api-response.js';
const NODEINFO_10 = 'http://nodeinfo.diaspora.software/ns/schema/1.0';
const NODEINFO_20 = 'http://nodeinfo.diaspora.software/ns/schema/2.0';
const NODEINFO_21 = 'http://nodeinfo.diaspora.software/ns/schema/2.1';
export class DefaultInstanceRouter {
    async getCustomEmojis(urlLike, software) {
        if (!software) {
            const data = await this.getSoftwareInfo(urlLike);
            software = data.software;
        }
        switch (software) {
            case KNOWN_SOFTWARE.FIREFISH:
            case KNOWN_SOFTWARE.ICESHRIMP:
            case KNOWN_SOFTWARE.MASTODON:
            case KNOWN_SOFTWARE.KMYBLUE: {
                const rex = await MastoJsWrapper.create(urlLike).lib.v1.customEmojis.list();
                return (rex?.map((o) => ({
                    shortCode: o.shortcode,
                    url: o.url,
                    staticUrl: o.staticUrl,
                    visibleInPicker: o.visibleInPicker,
                    category: o.category,
                    aliases: [],
                    tags: [],
                })) ?? []);
            }
            case KNOWN_SOFTWARE.SHARKEY:
            case KNOWN_SOFTWARE.MISSKEY:
            case KNOWN_SOFTWARE.CHERRYPICK: {
                const emojiFn = await MisskeyJsWrapper.create(urlLike).client.request('emojis', {});
                return emojiFn.emojis.map((o) => ({
                    shortCode: o.name,
                    url: o.url,
                    staticUrl: o.url,
                    visibleInPicker: true,
                    category: o.category,
                    aliases: o.aliases,
                    tags: [],
                }));
            }
            case KNOWN_SOFTWARE.PLEROMA:
            case KNOWN_SOFTWARE.AKKOMA: {
                const data = await new FetchWrapper(urlLike).get('/api/v1/custom_emojis');
                return data.map((o) => ({
                    shortCode: o.shortcode,
                    url: o.url,
                    staticUrl: o.static_url,
                    visibleInPicker: o.visible_in_picker,
                    aliases: [],
                    category: o.category,
                    tags: o.tags || [],
                }));
            }
            case KNOWN_SOFTWARE.GOTOSOCIAL: {
                const dt = await MegalodonGoToSocialWrapper.create(urlLike).client.getInstanceCustomEmojis();
                return dt.data.map((o) => ({
                    shortCode: o.shortcode,
                    url: o.url,
                    staticUrl: o.static_url,
                    visibleInPicker: o.visible_in_picker,
                    aliases: [],
                    category: o.category,
                    tags: [],
                }));
            }
            default: {
                throw new Error(`backend software incompatible for ${urlLike}`);
            }
        }
    }
    async getNodeInfo(urlLike) {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 5000);
        try {
            if (urlLike.startsWith('http://') || urlLike.startsWith('https://')) {
            }
            else {
                urlLike = 'https://' + urlLike;
            }
            const response = await fetch(urlLike + '/.well-known/nodeinfo', {
                signal: controller.signal,
            });
            if (!response.ok)
                return errorBuilder(response.statusText);
            const res = await response.json();
            const nodeType = res.links.find((l) => [NODEINFO_21, NODEINFO_20, NODEINFO_10].includes(l.rel));
            if (!nodeType)
                return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
            return nodeType.href;
        }
        catch (e) {
            throw new Error(`Failed to fetch nodeinfo from ${urlLike}`);
        }
        finally {
            clearTimeout(timeoutId);
        }
    }
    static getVersion(text) {
        const ex = /^([0-9]+.[0-9]+.[0-9]+)/;
        if (ex.test(text)) {
            return ex.exec(text)[1];
        }
        return null;
    }
    async getSoftware(nodeinfoUrl) {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 5000);
        try {
            const response = await fetch(nodeinfoUrl, {
                signal: controller.signal,
            });
            if (!response.ok)
                return errorBuilder(response.statusText);
            const res = await response.json();
            return {
                software: res?.software?.name,
                version: DefaultInstanceRouter.getVersion(res.version),
            };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
        finally {
            clearTimeout(timeoutId);
        }
    }
    async getSoftwareInfo(urlLike) {
        return identifyBackendSoftware(urlLike);
    }
    async getTranslation(id, lang) {
        throw new Error('Method not implemented.');
    }
    async getMastodonAccessToken(instanceUrl, code, clientId, clientSecret) {
        try {
            const res = await fetch(`https://${instanceUrl}/oauth/token`, {
                method: 'POST',
                body: JSON.stringify({
                    client_id: clientId,
                    client_secret: clientSecret,
                    redirect_uri: 'urn:ietf:wg:oauth:2.0:oob',
                    grant_type: 'authorization_code',
                    code,
                    scope: 'read write push follow',
                }),
                headers: {
                    'Content-Type': 'application/json',
                },
            });
            if (!res.ok) {
                return null;
            }
            const data = await res.json();
            return data?.access_token;
        }
        catch (e) {
            console.log('[ERROR]: obtaining mastodon token', e);
            return null;
        }
    }
    async verifyCredentials(urlLike, token) {
        const res = await fetch(`https://${urlLike}/api/v1/accounts/verify_credentials`, {
            method: 'GET',
            headers: {
                Authorization: `Bearer ${token}`,
                'Content-Type': 'application/json',
            },
        });
        if (!res.ok)
            return errorBuilder(res.statusText);
        return await res.json();
    }
}
//# sourceMappingURL=default.js.map