import { MisskeyJsWrapper } from '#/client/utils/api-wrappers.js';
import { getHumanReadableError } from '#/utils/errors.js';
import { errorBuilder } from '#/types/index.js';
import { ApiErrorCode } from '#/types/api-response.js';
export class MisskeyStatusesRouter {
    direct;
    client;
    constructor(forwarded) {
        this.direct = forwarded;
        this.client = MisskeyJsWrapper.create(forwarded.baseUrl, forwarded.token);
    }
    async create(dto) {
        try {
            return this.client.client.request('notes/create', {
                lang: dto.language,
                visibility: dto.misskeyVisibility,
                replyId: dto.inReplyToId,
                text: dto.status,
                visibleUserIds: dto.misskeyVisibility === 'specified'
                    ? dto.visibleUserIds || []
                    : undefined,
                fileIds: dto.mediaIds.length > 0 ? dto.mediaIds || [] : undefined,
                cw: dto.spoilerText || undefined,
                localOnly: false, // reactionAcceptance: null,
                poll: dto.poll || null,
                scheduledAt: null,
            });
        }
        catch (e) {
            throw new Error(getHumanReadableError(e));
        }
    }
    async delete(id) {
        try {
            await this.client.client.request('notes/delete', {
                noteId: id,
            });
            return { success: true, deleted: true };
        }
        catch (e) {
            if (e.code) {
                return { success: false, deleted: false };
            }
            return { success: false, deleted: false };
        }
    }
    async getPost(id) {
        return this.client.client.request('notes/show', { noteId: id });
    }
    async getReactions(postId) {
        return this.client.client.request('notes/reactions', {
            noteId: postId,
        });
    }
    async getReactionDetails(postId, reactionId) {
        try {
            const data = await this.client.client.request('notes/reactions', {
                noteId: postId,
                type: reactionId,
            });
            return { data };
        }
        catch (e) {
            return errorBuilder(ApiErrorCode.UNKNOWN_ERROR);
        }
    }
    async addReaction(postId, reactionId) {
        await this.client.client.request('notes/reactions/create', {
            noteId: postId,
            reaction: reactionId,
        });
        return { success: true, reacted: true, id: reactionId };
    }
    async removeReaction(postId, reactionId) {
        await this.client.client.request('notes/reactions/delete', {
            noteId: postId,
            reaction: reactionId,
        });
        return { success: false, reacted: false, id: reactionId };
    }
    async getState(id) {
        return this.client.client.request('notes/state', {
            noteId: id,
        });
    }
    async bookmark(id) {
        try {
            await this.client.client.request('notes/favorites/create', {
                noteId: id,
            });
            return { state: true };
        }
        catch (e) {
            throw new Error(getHumanReadableError(e));
        }
    }
    async unBookmark(id) {
        try {
            await this.client.client.request('notes/favorites/delete', {
                noteId: id,
            });
            return { state: false };
        }
        catch (e) {
            throw new Error(getHumanReadableError(e));
        }
    }
    async renotes(id) {
        return this.client.client.request('notes/renotes', {
            noteId: id,
        });
    }
    async favourite(id) {
        try {
            await this.direct.post('/api/notes/like', { noteId: id }, {});
            return { success: true, isFavourited: true };
        }
        catch (e) {
            throw new Error(getHumanReadableError(e));
        }
    }
    async like(id) {
        try {
            await this.direct.post('/api/notes/like', { noteId: id }, {});
            return { state: true };
        }
        catch (e) {
            throw new Error(getHumanReadableError(e));
        }
    }
    async removeLike(id) {
        throw new Error('Method not implemented.');
    }
    async renote(dto) {
        const data = await this.client.client.request('notes/create', dto);
        return {
            success: true,
            renoted: true,
            post: data,
        };
    }
    async unrenote(id) {
        await this.client.client.request('notes/unrenote', {
            noteId: id,
        });
        return { success: true, renoted: false };
    }
    async getPostContext(id, limit) {
        const parents = this.client.client.request('notes/conversation', {
            noteId: id,
            limit: limit || 40,
        });
        const children = this.client.client.request('notes/children', {
            noteId: id,
            showQuotes: false,
            limit: limit || 40,
        });
        return new Promise((resolve, reject) => {
            Promise.all([parents, children])
                .then(([_parents, _children]) => {
                resolve({
                    ancestors: _parents,
                    descendants: _children,
                });
            })
                .catch((e) => {
                throw new Error(getHumanReadableError(e));
            });
        });
    }
    async getLikedBy(id) {
        return {
            data: [],
        };
    }
    async getSharedBy(id, limit) {
        const renotes = await this.client.client.request('notes/renotes', {
            noteId: id,
            limit,
        });
        return {
            data: renotes,
        };
    }
    async getQuotedBy(id) {
        return { data: [] };
    }
}
//# sourceMappingURL=misskey.js.map