import axios from 'axios';
import objectAssignDeep from 'object-assign-deep';
import { RequestCanceledError } from '../cancel.js';
import { NO_REDIRECT, DEFAULT_SCOPE, DEFAULT_UA } from '../default.js';
import NotificationType, { UnknownNotificationTypeError } from '../notification.js';
import PixelfedNotificationType from './notification.js';
import Streaming from './web_socket.js';
var PixelfedAPI;
(function (PixelfedAPI) {
    class Client {
        static DEFAULT_SCOPE = DEFAULT_SCOPE;
        static DEFAULT_URL = 'https://pixelfed.social';
        static NO_REDIRECT = NO_REDIRECT;
        accessToken;
        baseUrl;
        abortController;
        constructor(baseUrl, accessToken = null, _userAgent = DEFAULT_UA) {
            this.accessToken = accessToken;
            this.baseUrl = baseUrl;
            this.abortController = new AbortController();
            axios.defaults.signal = this.abortController.signal;
        }
        async get(path, params = {}, headers = {}, pathIsFullyQualified = false) {
            let options = {
                params: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .get((pathIsFullyQualified ? '' : this.baseUrl) + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async put(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .put(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async putForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .putForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patch(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patch(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async patchForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .patchForm(this.baseUrl + path, params, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async post(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.post(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async postForm(path, params = {}, headers = {}) {
            let options = {
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios.postForm(this.baseUrl + path, params, options).then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        async del(path, params = {}, headers = {}) {
            let options = {
                data: params,
                headers: headers,
                maxContentLength: Infinity,
                maxBodyLength: Infinity
            };
            if (this.accessToken) {
                options = objectAssignDeep({}, options, {
                    headers: {
                        Authorization: `Bearer ${this.accessToken}`
                    }
                });
            }
            return axios
                .delete(this.baseUrl + path, options)
                .catch((err) => {
                if (axios.isCancel(err)) {
                    throw new RequestCanceledError(err.message);
                }
                else {
                    throw err;
                }
            })
                .then((resp) => {
                const res = {
                    data: resp.data,
                    status: resp.status,
                    statusText: resp.statusText,
                    headers: resp.headers
                };
                return res;
            });
        }
        cancel() {
            return this.abortController.abort();
        }
        socket() {
            return new Streaming();
        }
    }
    PixelfedAPI.Client = Client;
    let Converter;
    (function (Converter) {
        Converter.encodeNotificationType = (t) => {
            switch (t) {
                case NotificationType.Follow:
                    return PixelfedNotificationType.Follow;
                case NotificationType.Favourite:
                    return PixelfedNotificationType.Favourite;
                case NotificationType.Reblog:
                    return PixelfedNotificationType.Reblog;
                case NotificationType.Mention:
                    return PixelfedNotificationType.Mention;
                case NotificationType.FollowRequest:
                    return PixelfedNotificationType.FollowRequest;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.decodeNotificationType = (t) => {
            switch (t) {
                case PixelfedNotificationType.Follow:
                    return NotificationType.Follow;
                case PixelfedNotificationType.Favourite:
                    return NotificationType.Favourite;
                case PixelfedNotificationType.Mention:
                    return NotificationType.Mention;
                case PixelfedNotificationType.Reblog:
                    return NotificationType.Reblog;
                case PixelfedNotificationType.FollowRequest:
                    return NotificationType.FollowRequest;
                default:
                    return new UnknownNotificationTypeError();
            }
        };
        Converter.visibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
            }
        };
        Converter.encodeVisibility = (v) => {
            switch (v) {
                case 'public':
                    return 'public';
                case 'unlisted':
                    return 'unlisted';
                case 'private':
                    return 'private';
                case 'direct':
                    return 'direct';
                case 'local':
                    return 'public';
            }
        };
        Converter.account = (a) => ({
            id: a.id,
            username: a.username,
            acct: a.acct,
            display_name: a.display_name,
            locked: a.locked,
            discoverable: a.discoverable,
            group: null,
            noindex: null,
            suspended: null,
            limited: null,
            created_at: a.created_at,
            followers_count: a.followers_count,
            following_count: a.following_count,
            statuses_count: a.statuses_count,
            note: a.note,
            url: a.url,
            avatar: a.avatar,
            avatar_static: a.avatar_static,
            header: a.header,
            header_static: a.header_static,
            emojis: a.emojis,
            moved: null,
            fields: a.fields,
            bot: null,
            source: a.source
        });
        Converter.announcement = (a) => a;
        Converter.application = (a) => a;
        Converter.attachment = (a) => a;
        Converter.async_attachment = (a) => {
            if (a.url) {
                return {
                    id: a.id,
                    type: a.type,
                    url: a.url,
                    remote_url: a.remote_url,
                    preview_url: a.preview_url,
                    text_url: a.text_url,
                    meta: a.meta,
                    description: a.description,
                    blurhash: a.blurhash
                };
            }
            else {
                return a;
            }
        };
        Converter.context = (c) => ({
            ancestors: Array.isArray(c.ancestors) ? c.ancestors.map(a => Converter.status(a)) : [],
            descendants: Array.isArray(c.descendants) ? c.descendants.map(d => Converter.status(d)) : []
        });
        Converter.conversation = (c) => ({
            id: c.id,
            accounts: Array.isArray(c.accounts) ? c.accounts.map(a => Converter.account(a)) : [],
            last_status: c.last_status ? Converter.status(c.last_status) : null,
            unread: c.unread
        });
        Converter.emoji = (e) => e;
        Converter.field = (f) => f;
        Converter.filter = (f) => f;
        Converter.history = (h) => h;
        Converter.instance = (i) => ({
            uri: i.uri,
            title: i.title,
            description: i.description,
            email: i.email,
            version: i.version,
            thumbnail: i.thumbnail,
            urls: null,
            stats: Converter.stats(i.stats),
            languages: i.languages,
            registrations: i.registrations,
            approval_required: i.approval_required,
            configuration: {
                statuses: {
                    max_characters: i.configuration.statuses.max_characters,
                    max_media_attachments: i.configuration.statuses.max_media_attachments,
                    characters_reserved_per_url: i.configuration.statuses.characters_reserved_per_url
                },
                polls: {
                    max_options: i.configuration.polls.max_options,
                    max_characters_per_option: i.configuration.polls.max_characters_per_option,
                    min_expiration: i.configuration.polls.min_expiration,
                    max_expiration: i.configuration.polls.max_expiration
                }
            },
            contact_account: Converter.account(i.contact_account),
            rules: i.rules
        });
        Converter.marker = (m) => m;
        Converter.mention = (m) => m;
        Converter.notification = (n) => {
            const notificationType = Converter.decodeNotificationType(n.type);
            if (notificationType instanceof UnknownNotificationTypeError)
                return notificationType;
            if (n.status) {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    status: Converter.status(n.status),
                    type: notificationType
                };
            }
            else {
                return {
                    account: Converter.account(n.account),
                    created_at: n.created_at,
                    id: n.id,
                    type: notificationType
                };
            }
        };
        Converter.poll = (p) => p;
        Converter.poll_option = (p) => p;
        Converter.preferences = (p) => p;
        Converter.relationship = (r) => r;
        Converter.report = (r) => ({
            id: r.id,
            action_taken: r.action_taken,
            action_taken_at: r.action_taken_at,
            status_ids: r.status_ids,
            rule_ids: r.rule_ids,
            category: r.category,
            comment: r.comment,
            forwarded: r.forwarded,
            target_account: Converter.account(r.target_account)
        });
        Converter.results = (r) => ({
            accounts: Array.isArray(r.accounts) ? r.accounts.map(a => Converter.account(a)) : [],
            statuses: Array.isArray(r.statuses) ? r.statuses.map(s => Converter.status(s)) : [],
            hashtags: Array.isArray(r.hashtags) ? r.hashtags.map(h => Converter.tag(h)) : []
        });
        Converter.scheduled_status = (s) => s;
        Converter.source = (s) => s;
        Converter.stats = (s) => s;
        Converter.status = (s) => ({
            id: s.id,
            uri: s.uri,
            url: s.url,
            account: Converter.account(s.account),
            in_reply_to_id: s.in_reply_to_id,
            in_reply_to_account_id: s.in_reply_to_account_id,
            reblog: s.reblog ? Converter.status(s.reblog) : null,
            content: s.content,
            plain_content: null,
            created_at: s.created_at,
            edited_at: s.edited_at,
            emojis: Array.isArray(s.emojis) ? s.emojis.map(e => Converter.emoji(e)) : [],
            replies_count: s.replies_count,
            reblogs_count: s.reblogs_count,
            favourites_count: s.favourites_count,
            reblogged: s.reblogged,
            favourited: s.favourited,
            muted: s.muted,
            sensitive: s.sensitive,
            spoiler_text: s.spoiler_text,
            visibility: Converter.visibility(s.visibility),
            media_attachments: Array.isArray(s.media_attachments) ? s.media_attachments.map(m => Converter.attachment(m)) : [],
            mentions: Array.isArray(s.mentions) ? s.mentions.map(m => Converter.mention(m)) : [],
            tags: s.tags,
            card: null,
            poll: s.poll ? Converter.poll(s.poll) : null,
            application: s.application ? Converter.application(s.application) : null,
            language: s.language,
            pinned: false,
            emoji_reactions: [],
            bookmarked: s.bookmarked ? s.bookmarked : false,
            quote: null,
            quote_approval: {
                automatic: ['unsupported_policy'],
                manual: [],
                current_user: 'automatic'
            }
        });
        Converter.status_params = (s) => s;
        Converter.tag = (t) => t;
        Converter.token = (t) => t;
    })(Converter = PixelfedAPI.Converter || (PixelfedAPI.Converter = {}));
})(PixelfedAPI || (PixelfedAPI = {}));
export default PixelfedAPI;
