import { type HttpMetaParams } from "../../../interfaces/index.js";
import { type Account, type GroupedNotificationsResults, type NotificationGroupType } from "../../entities/v1/index.js";
import { type NotificationPolicy, type NotificationPolicyType } from "../../entities/v2/index.js";
import { type Method } from "../../method.js";
import { type Paginator } from "../../paginator.js";
import { type DefaultPaginationParams } from "../../resource.js";
export interface ListNotificationsParams extends DefaultPaginationParams {
    /** Types to include in the result. */
    readonly types?: readonly NotificationGroupType[] | null;
    /** Types to exclude from the results. */
    readonly excludeTypes?: readonly NotificationGroupType[] | null;
    /** Return only notifications received from the specified account. */
    readonly accountId?: string | null;
    /**
     * One of full (default) or partial_avatars. When set to partial_avatars,
     * some accounts will not be rendered in full in the returned accounts list but will be
     * instead returned in stripped-down form in the partial_accounts list. The most recent
     * account in a notification group is always rendered in full in the accounts attribute.
     */
    readonly expandAccounts?: "full" | "partial_avatars" | null;
    /**
     * Restrict which notification types can be grouped. Use this if there are notification types
     * for which your client does not support grouping. If omitted, the server will group notifications
     * of all types it supports (currently, favourite, follow and reblog). If you do not want any
     * notification grouping, use GET /api/v1/notifications instead. Notifications that would be
     * grouped if not for this parameter will instead be returned as individual single-notification
     * groups with a unique group_key that can be assumed to be of the form ungrouped-{notification_id}.
     *
     * Please note that neither the streaming API nor the individual notification APIs are aware of this
     * parameter and will always include a “proper” group_key that can be different from what is
     * returned here, meaning that you may have to ignore group_key for such notifications that you do
     * not want grouped and use ungrouped-{notification_id} instead for consistency.
     */
    readonly groupedTypes?: readonly NotificationGroupType[] | null;
    /** Whether to include notifications filtered by the user’s NotificationPolicy. Defaults to false. */
    readonly includeFiltered?: boolean | null;
}
export interface FetchUnreadCountParams {
    /** Maximum number of results to return. Defaults to 100 notifications. Max 1000 notifications. */
    readonly limit?: number | null;
    /** Types of notifications that should count towards unread notifications. */
    readonly types?: readonly NotificationGroupType[] | null;
    /** Types of notifications that should not count towards unread notifications. */
    readonly excludeTypes?: readonly NotificationGroupType[] | null;
    /** Only count unread notifications received from the specified account. */
    readonly accountId?: string | null;
    /** Restrict which notification types can be grouped. Use this if there are notification types for which your client does not support grouping. If omitted, the server will group notifications of all types it supports (currently, favourite, follow and reblog). If you do not want any notification grouping, use GET /api/v1/notifications/unread_count instead. */
    readonly groupedTypes?: readonly NotificationGroupType[] | null;
}
export interface UpdateNotificationPolicyParams {
    /** Whether to accept, filter or drop notifications from accounts the user is not following. drop will prevent creation of the notification object altogether (without preventing the underlying activity), filter will cause it to be marked as filtered, and accept will not affect its processing. */
    readonly forNotFollowing?: NotificationPolicyType | null;
    /** Whether to accept, filter or drop notifications from accounts that are not following the user. drop will prevent creation of the notification object altogether (without preventing the underlying activity), filter will cause it to be marked as filtered, and accept will not affect its processing. */
    readonly forNotFollowers?: NotificationPolicyType | null;
    /** Whether to accept, filter or drop notifications from accounts created in the past 30 days. drop will prevent creation of the notification object altogether (without preventing the underlying activity), filter will cause it to be marked as filtered, and accept will not affect its processing. */
    readonly forNewAccounts?: NotificationPolicyType | null;
    /** Whether to accept, filter or drop notifications from private mentions. drop will prevent creation of the notification object altogether (without preventing the underlying activity), filter will cause it to be marked as filtered, and accept will not affect its processing. Replies to private mentions initiated by the user, as well as accounts the user follows, are always allowed, regardless of this value. */
    readonly forPrivateMentions?: NotificationPolicyType | null;
    /** Whether to accept, filter or drop notifications from accounts that were limited by a moderator. drop will prevent creation of the notification object altogether (without preventing the underlying activity), filter will cause it to be marked as filtered, and accept will not affect its processing. */
    readonly forLimitedAccounts?: NotificationPolicyType | null;
}
export interface Notifications$SelectAccountsResource {
    fetch: Method<Account[]>;
}
export interface Notifications$SelectResource {
    accounts: Notifications$SelectAccountsResource;
    /**
     * View information about a specific notification group with a given group key.
     */
    fetch: Method<GroupedNotificationsResults>;
    /**
     * Dismiss a single notification group from the server.
     */
    dismiss: Method<void>;
}
export interface NotificationsUnreadCountResource {
    /**
     * Get the (capped) number of unread notification groups for the current user. A notification is
     * considered unread if it is more recent than the notifications read marker. Because the count
     * is dependant on the parameters, it is computed every time and is thus a relatively slow
     * operation (although faster than getting the full corresponding notifications), therefore the
     * number of returned notifications is capped.
     */
    fetch: Method<{
        count: number;
    }, FetchUnreadCountParams>;
}
export interface NotificationsPolicyResource {
    /**
     * Notifications filtering policy for the user.
     */
    fetch: Method<NotificationPolicy>;
    /**
     * Update the user's notifications filtering policy.
     */
    update: Method<NotificationPolicy, UpdateNotificationPolicyParams, HttpMetaParams<"json">>;
}
export interface NotificationsResource {
    /**
     * @param groupKey The group key of the notification group.
     */
    $select(groupKey: string): Notifications$SelectResource;
    unreadCount: NotificationsUnreadCountResource;
    policy: NotificationsPolicyResource;
    /**
     * Return grouped notifications concerning the user. This API returns Link headers containing links
     * to the next/previous page. However, the links can also be constructed dynamically using query
     * params and id values.
     *
     * Notifications of type favourite, follow or reblog with the same type and the same target made in
     * a similar timeframe are given a same group_key by the server, and querying this endpoint will
     * return aggregated notifications, with only one object per group_key. Other notification types may
     * be grouped in the future. The grouped_types parameter should be used by the client to explicitly
     * list the types it supports showing grouped notifications for.
     */
    list: Method<Paginator<GroupedNotificationsResults>, ListNotificationsParams>;
}
/** @deprecated Use NotificationsResource instead. */
export type NotificationRepository = NotificationsResource;
