"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseIntWithFallback = exports.dedupeStrs = exports.range = exports.chunkArray = exports.errHasMsg = exports.isErrnoException = exports.asyncFilter = exports.s32decode = exports.s32encode = exports.streamToBuffer = exports.flattenUint8Arrays = exports.bailableWait = exports.wait = exports.jitter = exports.noUndefinedVals = void 0;
exports.aggregateErrors = aggregateErrors;
exports.omit = omit;
const noUndefinedVals = (obj) => {
    Object.keys(obj).forEach((k) => {
        if (obj[k] === undefined) {
            delete obj[k];
        }
    });
    return obj;
};
exports.noUndefinedVals = noUndefinedVals;
function aggregateErrors(errors, message) {
    if (errors.length === 1) {
        return errors[0] instanceof Error
            ? errors[0]
            : new Error(message ?? stringifyError(errors[0]), { cause: errors[0] });
    }
    else {
        return new AggregateError(errors, message ?? `Multiple errors: ${errors.map(stringifyError).join('\n')}`);
    }
}
function stringifyError(reason) {
    if (reason instanceof Error) {
        return reason.message;
    }
    return String(reason);
}
function omit(src, rejectedKeys) {
    // Hot path
    if (!src)
        return src;
    const dst = {};
    const srcKeys = Object.keys(src);
    for (let i = 0; i < srcKeys.length; i++) {
        const key = srcKeys[i];
        if (!rejectedKeys.includes(key)) {
            dst[key] = src[key];
        }
    }
    return dst;
}
const jitter = (maxMs) => {
    return Math.round((Math.random() - 0.5) * maxMs * 2);
};
exports.jitter = jitter;
const wait = (ms) => {
    return new Promise((res) => setTimeout(res, ms));
};
exports.wait = wait;
const bailableWait = (ms) => {
    let bail;
    const waitPromise = new Promise((res) => {
        const timeout = setTimeout(res, ms);
        bail = () => {
            clearTimeout(timeout);
            res();
        };
    });
    return { bail, wait: () => waitPromise };
};
exports.bailableWait = bailableWait;
const flattenUint8Arrays = (arrs) => {
    const length = arrs.reduce((acc, cur) => {
        return acc + cur.length;
    }, 0);
    const flattened = new Uint8Array(length);
    let offset = 0;
    arrs.forEach((arr) => {
        flattened.set(arr, offset);
        offset += arr.length;
    });
    return flattened;
};
exports.flattenUint8Arrays = flattenUint8Arrays;
const streamToBuffer = async (stream) => {
    const arrays = [];
    for await (const chunk of stream) {
        arrays.push(chunk);
    }
    return (0, exports.flattenUint8Arrays)(arrays);
};
exports.streamToBuffer = streamToBuffer;
const S32_CHAR = '234567abcdefghijklmnopqrstuvwxyz';
const s32encode = (i) => {
    let s = '';
    while (i) {
        const c = i % 32;
        i = Math.floor(i / 32);
        s = S32_CHAR.charAt(c) + s;
    }
    return s;
};
exports.s32encode = s32encode;
const s32decode = (s) => {
    let i = 0;
    for (const c of s) {
        i = i * 32 + S32_CHAR.indexOf(c);
    }
    return i;
};
exports.s32decode = s32decode;
const asyncFilter = async (arr, fn) => {
    const results = await Promise.all(arr.map((t) => fn(t)));
    return arr.filter((_, i) => results[i]);
};
exports.asyncFilter = asyncFilter;
const isErrnoException = (err) => {
    return !!err && err['code'];
};
exports.isErrnoException = isErrnoException;
const errHasMsg = (err, msg) => {
    return !!err && typeof err === 'object' && err['message'] === msg;
};
exports.errHasMsg = errHasMsg;
const chunkArray = (arr, chunkSize) => {
    return arr.reduce((acc, cur, i) => {
        const chunkI = Math.floor(i / chunkSize);
        if (!acc[chunkI]) {
            acc[chunkI] = [];
        }
        acc[chunkI].push(cur);
        return acc;
    }, []);
};
exports.chunkArray = chunkArray;
const range = (num) => {
    const nums = [];
    for (let i = 0; i < num; i++) {
        nums.push(i);
    }
    return nums;
};
exports.range = range;
const dedupeStrs = (strs) => {
    return [...new Set(strs)];
};
exports.dedupeStrs = dedupeStrs;
const parseIntWithFallback = (value, fallback) => {
    const parsed = parseInt(value || '', 10);
    return isNaN(parsed) ? fallback : parsed;
};
exports.parseIntWithFallback = parseIntWithFallback;
//# sourceMappingURL=util.js.map