import { defaultResultPage, DriverService, KeyExtractorUtil, KNOWN_SOFTWARE, PostParser, } from '@dhaaga/bridge';
import { TimelineFetchMode } from '@dhaaga/core';
import { queryOptions } from '@tanstack/react-query';
const TIMELINE_STATUS_LIMIT = 20;
export function feedUnifiedQueryOptions(client, driver, server, acctIdentifier, { type, query, opts, maxId, minId, sessionId }) {
    const _id = query?.id;
    const _query = {
        ...opts,
        limit: TIMELINE_STATUS_LIMIT,
        sinceId: minId,
        untilId: maxId === null ? undefined : maxId,
        maxId,
        minId, // quirks
        userId: _id,
    };
    function generateMaxId(data, maxId) {
        if (maxId)
            return maxId;
        if (DriverService.supportsAtProto(driver)) {
            if (data.posts !== undefined)
                return data.cursor;
            const _payload = data;
            return _payload.data.cursor;
        }
        else {
            return data[data.length - 1]?.id;
        }
    }
    function getPageFromResult(result) {
        if (!result) {
            console.log('[WARN]: this timeline failed to load...');
            return defaultResultPage;
        }
        return KeyExtractorUtil.getPage(result, (o) => PostParser.parse(o, driver, server));
    }
    function outputSchemaToResultPage(data) {
        return {
            items: PostParser.parse(data.feed, driver, server),
            maxId: data.cursor === undefined ? null : data.cursor,
            minId: null,
        };
    }
    function generateFeedBatch(data) {
        let _feed = [];
        if (DriverService.supportsAtProto(driver)) {
            if (data.posts) {
                _feed = data.posts;
            }
            else {
                const _payload = data;
                _feed = _payload.data.feed;
            }
        }
        else {
            _feed = data.posts ? data.posts : data;
        }
        try {
            return PostParser.parse(_feed, driver, server);
        }
        catch (e) {
            console.log('[ERROR]: failed to convert posts', e);
            return [];
        }
    }
    function createResultBatch(data, maxId) {
        return {
            items: generateFeedBatch(data),
            maxId: generateMaxId(data, maxId),
            minId: null,
        };
    }
    async function api() {
        if (client === null)
            return defaultResultPage;
        switch (type) {
            case TimelineFetchMode.IDLE:
                return defaultResultPage;
            case TimelineFetchMode.HOME: {
                const result = await client.timelines.home(_query);
                return createResultBatch(result);
            }
            case TimelineFetchMode.LOCAL: {
                const result = await client.timelines.public({
                    ..._query,
                    local: true, // Pleroma/Akkoma thing
                    withMuted: DriverService.supportsPleromaApi(driver)
                        ? true
                        : undefined,
                    withRenotes: !opts?.excludeReblogs,
                    withReplies: !opts?.excludeReplies,
                });
                console.log(result);
                return createResultBatch(result);
            }
            case TimelineFetchMode.HASHTAG: {
                const result = await client.timelines.hashtag(_id, _query);
                return createResultBatch(result);
            }
            case TimelineFetchMode.LIST: {
                const result = await client.timelines.list(_id, _query);
                return createResultBatch(result);
            }
            case TimelineFetchMode.USER: {
                if (!_query || _query.userId === undefined)
                    throw new Error('missing userId');
                const result = await client.accounts.statuses(_id, _query);
                if (result.isErr())
                    return defaultResultPage;
                return createResultBatch(result);
            }
            case TimelineFetchMode.SOCIAL: {
                const result = await client.timelines.public({
                    ..._query,
                    social: true,
                });
                return createResultBatch(result);
            }
            case TimelineFetchMode.BUBBLE: {
                if (DriverService.supportsPleromaApi(driver)) {
                    const result = await client.timelines.bubble(_query);
                    return {
                        items: PostParser.parse(result, driver, server),
                        maxId: generateMaxId(result),
                        minId: null,
                    };
                }
                else if (driver === KNOWN_SOFTWARE.SHARKEY) {
                    const result = await client.timelines.bubble(_query);
                    return {
                        items: PostParser.parse(result, driver, server),
                        maxId: generateMaxId(result),
                        minId: null,
                    };
                }
                else {
                    return defaultResultPage;
                }
            }
            case TimelineFetchMode.FEDERATED: {
                const result = await client.timelines.public(_query);
                return getPageFromResult(result);
            }
            case TimelineFetchMode.BOOKMARKS: {
                const { data, error } = await client.accounts.bookmarks(_query);
                if (error)
                    return defaultResultPage;
                return createResultBatch(data.data, data?.maxId);
            }
            case TimelineFetchMode.FEED: {
                const { data, error } = await client.timelines.feed({
                    limit: TIMELINE_STATUS_LIMIT,
                    cursor: maxId === null ? undefined : maxId,
                    feed: query.id,
                });
                if (error)
                    return defaultResultPage;
                return outputSchemaToResultPage(data);
            }
            case TimelineFetchMode.LIKES: {
                if (DriverService.supportsAtProto(driver)) {
                    const { data, error } = await client.accounts.atProtoLikes(acctIdentifier, {
                        limit: TIMELINE_STATUS_LIMIT,
                        cursor: _query.maxId === null ? undefined : _query.maxId,
                    });
                    if (error)
                        return defaultResultPage;
                    return {
                        items: PostParser.parse(data.feed, driver, server),
                        maxId: data.cursor === undefined ? null : data.cursor,
                        minId: null,
                    };
                }
                const { data, error } = await client.accounts.likes(_query);
                if (error)
                    return defaultResultPage;
                return createResultBatch(data.data, data.maxId);
            }
            default:
                return defaultResultPage;
        }
    }
    return queryOptions({
        queryKey: ['feed/unified', type, _id, _query, maxId, minId, sessionId],
        queryFn: api,
        enabled: !!client && type !== TimelineFetchMode.IDLE,
        initialData: defaultResultPage,
    });
}
//# sourceMappingURL=feed.js.map