class ColumnInterface {
    ref;
    constructor(type) {
        this.ref = {
            pk: false,
            fk: false,
            fkTo: null,
            notNull: false,
            default: undefined,
            type,
        };
    }
    notNull() {
        this.ref.notNull = true;
        return this;
    }
    pk() {
        this.ref.pk = true;
        return this;
    }
    fk(to) {
        this.ref.fk = true;
        this.ref.fkTo = to;
        return this;
    }
    default(val) {
        this.ref.default = val.toString();
        return this;
    }
}
/**
 * helper function to generate sql for "CREATE TABLE"
 *
 * NOTE: please read design assumptions in _migrator.ts
 * @param name of the table
 * @param schema of the table
 */
function createTable(name, schema) {
    let sql = `CREATE TABLE IF NOT EXISTS ${name} (`;
    let pkList = [];
    let fKList = [];
    for (const [key, val] of Object.entries(schema)) {
        if (val.ref.pk) {
            pkList.push({
                key,
                type: val.ref.type,
            });
        }
        if (val.ref.fk) {
            fKList.push({ key: key, to: val.ref.fkTo });
        }
    }
    // handle primary keys
    for (const pk of pkList) {
        sql += `${pk.key} INTEGER PRIMARY KEY, `;
    }
    for (const [key, val] of Object.entries(schema)) {
        if (val.ref.pk || val.ref.fk)
            continue;
        switch (val.ref.type) {
            case 'text': {
                sql += `${key} TEXT`;
                break;
            }
            case 'int': {
                sql += `${key} INTEGER`;
                break;
            }
            case 'float': {
                sql += `${key} REAL`;
                break;
            }
            case 'blob': {
                sql += `${key} BLOB`;
                break;
            }
        }
        if (val.ref.notNull) {
            sql += ` NOT NULL`;
        }
        if (val.ref.default !== undefined) {
            sql += ` DEFAULT ${val.ref.default}`;
        }
        sql += ', ';
    }
    // handle foreign keys
    for (const fk of fKList) {
        sql += `${fk.key} INTEGER, `;
    }
    sql += `createdAt TEXT DEFAULT CURRENT_TIMESTAMP, `;
    sql += `updatedAt TEXT DEFAULT CURRENT_TIMESTAMP, `;
    for (const fk of fKList) {
        sql += `FOREIGN KEY (${fk.key}) REFERENCES ${fk.to} (id) ON DELETE CASCADE, `;
    }
    sql = sql.trimEnd().replace(/,$/, ''); // Trim spaces and remove trailing comma
    sql += `) STRICT;`;
    return sql;
}
function addColumn(tableName, columnName, dataType, notNull = false, defaultValue = null) {
    let type = dataType.toUpperCase();
    let q = `ALTER TABLE \'${tableName}\' ADD COLUMN \`${columnName}\` ${type}`;
    if (notNull)
        q += ` NOT NULL`;
    if (defaultValue)
        q += ` DEFAULT ${defaultValue.toString()}`;
    return q + ';';
}
function removeColumn(tableName, columnName) {
    return `ALTER TABLE \'${tableName}\' DROP COLUMN \'${columnName}\';`;
}
function dropTable(name) {
    return `DROP TABLE IF EXISTS ${name};`;
}
const migrator = {
    text: () => new ColumnInterface('text'),
    int: () => new ColumnInterface('int'),
    float: () => new ColumnInterface('float'),
    blob: () => new ColumnInterface('blob'),
};
export { migrator, createTable, dropTable, addColumn, removeColumn };
